import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, Link } from '@inertiajs/react';
import { formatCurrency, formatDate } from '@/utils';

export default function Show({ tenant }) {
    const activeLease = tenant.leases?.find((l) => l.status === 'active');

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Tenant Details</h2>}>
            <Head title={tenant.name} />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-8 space-y-6">
                    {/* Tenant Info Card */}
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium">{tenant.name}</h3>
                                <div className="space-x-2">
                                    <Link href={route('tenants.edit', tenant.id)} className="inline-flex items-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white hover:bg-indigo-700">Edit</Link>
                                    <Link href={route('tenants.index')} className="inline-flex items-center rounded-md bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300">Back</Link>
                                </div>
                            </div>

                            <div className="flex gap-6">
                                {tenant.photo && (
                                    <div className="flex-shrink-0">
                                        <img src={`/storage/${tenant.photo}`} alt={tenant.name} className="h-24 w-24 rounded-full object-cover" />
                                    </div>
                                )}
                                <dl className="grid flex-1 grid-cols-1 gap-x-4 gap-y-4 sm:grid-cols-2 lg:grid-cols-3">
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Name</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{tenant.name}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Father Name</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{tenant.father_name || '—'}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">CNIC</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{tenant.cnic || '—'}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Phone</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{tenant.phone}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Email</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{tenant.email || '—'}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Address</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{tenant.address || '—'}</dd>
                                    </div>
                                </dl>
                            </div>
                        </div>
                    </div>

                    {/* Active Lease */}
                    {activeLease && (
                        <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                            <div className="p-6 text-gray-900">
                                <h3 className="text-lg font-medium mb-4">Active Lease</h3>
                                <dl className="grid grid-cols-1 gap-x-4 gap-y-4 sm:grid-cols-3">
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Shop</dt>
                                        <dd className="mt-1 text-sm text-gray-900">
                                            <Link href={route('shops.show', activeLease.shop?.id)} className="text-blue-600 hover:text-blue-900">
                                                {activeLease.shop?.shop_number}
                                            </Link>
                                        </dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Start Date</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{formatDate(activeLease.start_date)}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Monthly Rent</dt>
                                        <dd className="mt-1 text-sm text-gray-900">{formatCurrency(activeLease.rent)}</dd>
                                    </div>
                                </dl>
                            </div>
                        </div>
                    )}

                    {/* Invoices */}
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            <h3 className="text-lg font-medium mb-4">Invoices</h3>
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Invoice #</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Month</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Rent</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Electricity</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Total</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Paid</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Status</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 bg-white">
                                        {tenant.leases?.flatMap((lease) =>
                                            (lease.invoices || []).map((invoice) => (
                                                <tr key={invoice.id} className="hover:bg-gray-50">
                                                    <td className="whitespace-nowrap px-6 py-4 text-sm font-medium text-gray-900">{invoice.invoice_number}</td>
                                                    <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{invoice.month}</td>
                                                    <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{formatCurrency(invoice.rent_amount)}</td>
                                                    <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{formatCurrency(invoice.electricity_amount)}</td>
                                                    <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{formatCurrency(invoice.total_amount)}</td>
                                                    <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">
                                                        {formatCurrency(
                                                            (invoice.payments || []).reduce((sum, p) => sum + Number(p.amount), 0)
                                                        )}
                                                    </td>
                                                    <td className="whitespace-nowrap px-6 py-4 text-sm">
                                                        <span className={`inline-flex rounded-full px-2 text-xs font-semibold leading-5 ${
                                                            invoice.status === 'paid'
                                                                ? 'bg-green-100 text-green-800'
                                                                : invoice.status === 'partial'
                                                                    ? 'bg-yellow-100 text-yellow-800'
                                                                    : 'bg-red-100 text-red-800'
                                                        }`}>
                                                            {invoice.status}
                                                        </span>
                                                    </td>
                                                </tr>
                                            ))
                                        )}
                                        {(!tenant.leases || tenant.leases.flatMap((l) => l.invoices || []).length === 0) && (
                                            <tr>
                                                <td colSpan="7" className="px-6 py-4 text-center text-sm text-gray-500">No invoices found.</td>
                                            </tr>
                                        )}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
}
