import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import ConfirmDeleteModal from '@/Components/ConfirmDeleteModal';
import FormModal from '@/Components/FormModal';
import { Head, Link, useForm, router, usePage } from '@inertiajs/react';
import { useState, useEffect } from 'react';

const inputClass = "mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 text-sm";

const emptyTenant = {
    name: '',
    father_name: '',
    cnic: '',
    phone: '',
    email: '',
    address: '',
    photo: null,
};

function TenantFormFields({ form }) {
    return (
        <>
            {/* Name */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Name <span className="text-red-500">*</span></label>
                <input type="text" value={form.data.name} onChange={e => form.setData('name', e.target.value)} className={inputClass} />
                {form.errors.name && <p className="mt-1 text-sm text-red-600">{form.errors.name}</p>}
            </div>

            {/* Father Name */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Father Name</label>
                <input type="text" value={form.data.father_name} onChange={e => form.setData('father_name', e.target.value)} className={inputClass} />
                {form.errors.father_name && <p className="mt-1 text-sm text-red-600">{form.errors.father_name}</p>}
            </div>

            {/* CNIC */}
            <div>
                <label className="block text-sm font-medium text-gray-700">CNIC</label>
                <input type="text" value={form.data.cnic} onChange={e => form.setData('cnic', e.target.value)} placeholder="00000-0000000-0" className={inputClass} />
                {form.errors.cnic && <p className="mt-1 text-sm text-red-600">{form.errors.cnic}</p>}
            </div>

            {/* Phone + Email */}
            <div className="grid grid-cols-2 gap-4">
                <div>
                    <label className="block text-sm font-medium text-gray-700">Phone <span className="text-red-500">*</span></label>
                    <input type="text" value={form.data.phone} onChange={e => form.setData('phone', e.target.value)} className={inputClass} />
                    {form.errors.phone && <p className="mt-1 text-sm text-red-600">{form.errors.phone}</p>}
                </div>
                <div>
                    <label className="block text-sm font-medium text-gray-700">Email</label>
                    <input type="email" value={form.data.email} onChange={e => form.setData('email', e.target.value)} className={inputClass} />
                    {form.errors.email && <p className="mt-1 text-sm text-red-600">{form.errors.email}</p>}
                </div>
            </div>

            {/* Address */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Address</label>
                <textarea rows={2} value={form.data.address} onChange={e => form.setData('address', e.target.value)} className={inputClass} />
                {form.errors.address && <p className="mt-1 text-sm text-red-600">{form.errors.address}</p>}
            </div>

            {/* Photo */}
            <div>
                <label className="block text-sm font-medium text-gray-700">Photo</label>
                <input
                    type="file"
                    accept="image/*"
                    onChange={e => form.setData('photo', e.target.files[0] || null)}
                    className="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:rounded-lg file:border-0 file:bg-blue-50 file:px-4 file:py-2 file:text-sm file:font-medium file:text-blue-700 hover:file:bg-blue-100"
                />
                {form.errors.photo && <p className="mt-1 text-sm text-red-600">{form.errors.photo}</p>}
            </div>
        </>
    );
}

export default function Index({ tenants, filters }) {
    const { auth } = usePage().props;
    const [deleteId, setDeleteId] = useState(null);
    const [deleting, setDeleting] = useState(false);
    const [showAddModal, setShowAddModal] = useState(false);
    const [editingTenant, setEditingTenant] = useState(null);

    const addForm = useForm({ ...emptyTenant });
    const editForm = useForm({ ...emptyTenant });

    const applyFilters = (newFilters) => {
        router.get(route('tenants.index'), { ...filters, ...newFilters }, { preserveState: true, replace: true });
    };

    useEffect(() => {
        if (editingTenant) {
            editForm.setData({
                name: editingTenant.name || '',
                father_name: editingTenant.father_name || '',
                cnic: editingTenant.cnic || '',
                phone: editingTenant.phone || '',
                email: editingTenant.email || '',
                address: editingTenant.address || '',
                photo: null,
            });
        }
    }, [editingTenant]);

    const handleDelete = () => {
        setDeleting(true);
        router.delete(route('tenants.destroy', deleteId), {
            onFinish: () => { setDeleting(false); setDeleteId(null); },
        });
    };

    const handleAdd = (e) => {
        e.preventDefault();
        addForm.post(route('tenants.store'), {
            forceFormData: true,
            onSuccess: () => { setShowAddModal(false); addForm.reset(); },
        });
    };

    const handleEdit = (e) => {
        e.preventDefault();
        editForm.post(route('tenants.update', editingTenant.id), {
            forceFormData: true,
            _method: 'put',
            onSuccess: () => setEditingTenant(null),
        });
    };

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Tenants</h2>}>
            <Head title="Tenants" />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-8">
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            {/* Header */}
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium">All Tenants</h3>
                                <button
                                    type="button"
                                    onClick={() => setShowAddModal(true)}
                                    className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                                >
                                    + Add Tenant
                                </button>
                            </div>

                            {/* Filters */}
                            <div className="mb-6">
                                <input
                                    type="text"
                                    placeholder="Search by name, CNIC, or phone..."
                                    defaultValue={filters.search}
                                    onChange={(e) => applyFilters({ search: e.target.value })}
                                    className="w-full max-w-md rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                />
                            </div>

                            {/* Table */}
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Name</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Father Name</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">CNIC</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Phone</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Active Shop</th>
                                            <th className="px-6 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 bg-white">
                                        {tenants.data.map((tenant) => (
                                            <tr key={tenant.id} className="hover:bg-gray-50">
                                                <td className="whitespace-nowrap px-6 py-4 text-sm font-medium text-gray-900">{tenant.name}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{tenant.father_name || '—'}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{tenant.cnic || '—'}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">{tenant.phone}</td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm text-gray-500">
                                                    {tenant.active_lease?.shop?.shop_number || '—'}
                                                </td>
                                                <td className="whitespace-nowrap px-6 py-4 text-sm space-x-2">
                                                    <Link href={route('tenants.show', tenant.id)} className="text-blue-600 hover:text-blue-900">View</Link>
                                                    {auth.user.is_admin && (
                                                    <button type="button" onClick={() => setEditingTenant(tenant)} className="text-indigo-600 hover:text-indigo-900">Edit</button>
                                                    )}
                                                    {auth.user.is_admin && (
                                                    <button onClick={() => setDeleteId(tenant.id)} className="text-red-600 hover:text-red-900">Delete</button>
                                                    )}
                                                </td>
                                            </tr>
                                        ))}
                                        {tenants.data.length === 0 && (
                                            <tr>
                                                <td colSpan="6" className="px-6 py-4 text-center text-sm text-gray-500">No tenants found.</td>
                                            </tr>
                                        )}
                                    </tbody>
                                </table>
                            </div>

                            {/* Pagination */}
                            {tenants.links && tenants.links.length > 3 && (
                                <div className="flex flex-wrap items-center gap-1 mt-4">
                                    {tenants.links.map((link, i) => (
                                        link.url ? (
                                            <Link key={i} href={link.url} className={`px-3 py-1 text-sm rounded ${link.active ? 'bg-blue-600 text-white' : 'bg-white border hover:bg-gray-50'}`}
                                                dangerouslySetInnerHTML={{ __html: link.label }} preserveState />
                                        ) : (
                                            <span key={i} className="px-3 py-1 text-sm text-gray-400" dangerouslySetInnerHTML={{ __html: link.label }} />
                                        )
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>

            {/* Delete Modal */}
            <ConfirmDeleteModal
                show={deleteId !== null}
                onClose={() => setDeleteId(null)}
                onConfirm={handleDelete}
                title="Delete Tenant"
                message="Are you sure you want to delete this tenant? All associated leases will be affected."
                processing={deleting}
            />

            {/* Add Modal */}
            <FormModal show={showAddModal} onClose={() => setShowAddModal(false)} title="Add New Tenant">
                <form onSubmit={handleAdd} className="space-y-4">
                    <TenantFormFields form={addForm} />
                    <div className="flex justify-end gap-3 pt-4 border-t">
                        <button type="button" onClick={() => setShowAddModal(false)} className="inline-flex items-center rounded-md bg-gray-100 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-300">
                            Cancel
                        </button>
                        <button type="submit" disabled={addForm.processing} className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white shadow-sm hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50">
                            Save Tenant
                        </button>
                    </div>
                </form>
            </FormModal>

            {/* Edit Modal */}
            <FormModal show={editingTenant !== null} onClose={() => setEditingTenant(null)} title="Edit Tenant">
                <form onSubmit={handleEdit} className="space-y-4">
                    <TenantFormFields form={editForm} />
                    <div className="flex justify-end gap-3 pt-4 border-t">
                        <button type="button" onClick={() => setEditingTenant(null)} className="inline-flex items-center rounded-md bg-gray-100 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-200 focus:outline-none focus:ring-2 focus:ring-gray-300">
                            Cancel
                        </button>
                        <button type="submit" disabled={editForm.processing} className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white shadow-sm hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50">
                            Update Tenant
                        </button>
                    </div>
                </form>
            </FormModal>
        </AuthenticatedLayout>
    );
}
