import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, useForm, router } from '@inertiajs/react';
import { useState } from 'react';
import FormModal from '@/Components/FormModal';
import ConfirmDeleteModal from '@/Components/ConfirmDeleteModal';

export default function Index({ settings, categories }) {
    const [activeTab, setActiveTab] = useState('business');
    const [showCategoryModal, setShowCategoryModal] = useState(false);
    const [editingCategory, setEditingCategory] = useState(null);
    const [deleteCategory, setDeleteCategory] = useState(null);

    // Settings form
    const settingsForm = useForm({
        business_name: settings.business_name || '',
        business_address: settings.business_address || '',
        business_phone: settings.business_phone || '',
        business_logo: null,
        favicon: null,
        default_electricity_rate: settings.default_electricity_rate || '',
        invoice_prefix: settings.invoice_prefix || '',
        voucher_prefix: settings.voucher_prefix || '',
    });

    // Category form
    const categoryForm = useForm({
        name: '',
        type: 'expense',
    });

    const handleSettingsSubmit = (e) => {
        e.preventDefault();
        settingsForm.post(route('settings.update'), { forceFormData: true });
    };

    const openCreateCategory = () => {
        setEditingCategory(null);
        categoryForm.reset();
        categoryForm.clearErrors();
        setShowCategoryModal(true);
    };

    const openEditCategory = (category) => {
        setEditingCategory(category);
        categoryForm.setData({
            name: category.name,
            type: category.type,
        });
        categoryForm.clearErrors();
        setShowCategoryModal(true);
    };

    const handleCategorySubmit = (e) => {
        e.preventDefault();
        if (editingCategory) {
            categoryForm.put(route('settings.categories.update', editingCategory.id), {
                onSuccess: () => setShowCategoryModal(false),
            });
        } else {
            categoryForm.post(route('settings.categories.store'), {
                onSuccess: () => setShowCategoryModal(false),
            });
        }
    };

    const handleDeleteCategory = () => {
        router.delete(route('settings.categories.destroy', deleteCategory.id), {
            onSuccess: () => setDeleteCategory(null),
        });
    };

    const incomeCategories = categories.filter(c => c.type === 'income');
    const expenseCategories = categories.filter(c => c.type === 'expense');

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Settings</h2>}>
            <Head title="Settings" />

            <div className="py-12">
                <div className="mx-auto max-w-4xl sm:px-6 lg:px-8">
                    {/* Tabs */}
                    <div className="bg-white shadow-sm sm:rounded-lg overflow-hidden">
                        <div className="border-b border-gray-200">
                            <div className="flex">
                                <button
                                    onClick={() => setActiveTab('business')}
                                    className={`px-6 py-4 text-sm font-medium border-b-2 transition-colors ${
                                        activeTab === 'business'
                                            ? 'border-blue-600 text-blue-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    Business Settings
                                </button>
                                <button
                                    onClick={() => setActiveTab('categories')}
                                    className={`px-6 py-4 text-sm font-medium border-b-2 transition-colors ${
                                        activeTab === 'categories'
                                            ? 'border-blue-600 text-blue-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    Cash Book Categories
                                </button>
                            </div>
                        </div>

                        {/* Business Settings Tab */}
                        {activeTab === 'business' && (
                            <div className="p-6">
                                <h3 className="text-lg font-medium mb-6 text-gray-900">Business Information</h3>

                                <form onSubmit={handleSettingsSubmit} className="space-y-6">
                                    {/* Business Name */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Business Name</label>
                                        <input
                                            type="text"
                                            value={settingsForm.data.business_name}
                                            onChange={(e) => settingsForm.setData('business_name', e.target.value)}
                                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                        />
                                        {settingsForm.errors.business_name && <p className="mt-1 text-sm text-red-600">{settingsForm.errors.business_name}</p>}
                                    </div>

                                    {/* Business Address */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Business Address</label>
                                        <textarea
                                            value={settingsForm.data.business_address}
                                            onChange={(e) => settingsForm.setData('business_address', e.target.value)}
                                            rows={3}
                                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                        />
                                        {settingsForm.errors.business_address && <p className="mt-1 text-sm text-red-600">{settingsForm.errors.business_address}</p>}
                                    </div>

                                    {/* Business Phone */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Business Phone</label>
                                        <input
                                            type="text"
                                            value={settingsForm.data.business_phone}
                                            onChange={(e) => settingsForm.setData('business_phone', e.target.value)}
                                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                        />
                                        {settingsForm.errors.business_phone && <p className="mt-1 text-sm text-red-600">{settingsForm.errors.business_phone}</p>}
                                    </div>

                                    {/* Business Logo */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Business Logo</label>
                                        {settings.business_logo && (
                                            <div className="mb-2">
                                                <img src={`/storage/${settings.business_logo}`} alt="Business Logo" className="h-16 rounded border border-gray-200" />
                                            </div>
                                        )}
                                        <input
                                            type="file"
                                            accept="image/*"
                                            onChange={(e) => settingsForm.setData('business_logo', e.target.files[0])}
                                            className="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:rounded-md file:border-0 file:bg-blue-50 file:px-4 file:py-2 file:text-sm file:font-medium file:text-blue-700 hover:file:bg-blue-100"
                                        />
                                        {settingsForm.errors.business_logo && <p className="mt-1 text-sm text-red-600">{settingsForm.errors.business_logo}</p>}
                                    </div>

                                    {/* Favicon */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Favicon (Browser Tab Icon)</label>
                                        <p className="text-xs text-gray-500 mb-2">Recommended: 16x16 or 32x32 pixels, .ico or .png format</p>
                                        {settings.favicon && (
                                            <div className="mb-2">
                                                <img src={`/storage/${settings.favicon}`} alt="Favicon" className="h-8 w-8 rounded border border-gray-200" />
                                            </div>
                                        )}
                                        <input
                                            type="file"
                                            accept="image/x-icon,image/png,image/jpeg"
                                            onChange={(e) => settingsForm.setData('favicon', e.target.files[0])}
                                            className="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:rounded-md file:border-0 file:bg-blue-50 file:px-4 file:py-2 file:text-sm file:font-medium file:text-blue-700 hover:file:bg-blue-100"
                                        />
                                        {settingsForm.errors.favicon && <p className="mt-1 text-sm text-red-600">{settingsForm.errors.favicon}</p>}
                                    </div>

                                    {/* Default Electricity Rate */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Default Electricity Rate (PKR/unit)</label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            min="0"
                                            value={settingsForm.data.default_electricity_rate}
                                            onChange={(e) => settingsForm.setData('default_electricity_rate', e.target.value)}
                                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                        />
                                        {settingsForm.errors.default_electricity_rate && <p className="mt-1 text-sm text-red-600">{settingsForm.errors.default_electricity_rate}</p>}
                                    </div>

                                    {/* Invoice Prefix */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Invoice Prefix</label>
                                        <input
                                            type="text"
                                            value={settingsForm.data.invoice_prefix}
                                            onChange={(e) => settingsForm.setData('invoice_prefix', e.target.value)}
                                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                            placeholder="e.g. INV"
                                        />
                                        {settingsForm.errors.invoice_prefix && <p className="mt-1 text-sm text-red-600">{settingsForm.errors.invoice_prefix}</p>}
                                    </div>

                                    {/* Voucher Prefix */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700">Voucher Prefix</label>
                                        <input
                                            type="text"
                                            value={settingsForm.data.voucher_prefix}
                                            onChange={(e) => settingsForm.setData('voucher_prefix', e.target.value)}
                                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                            placeholder="e.g. OV"
                                        />
                                        {settingsForm.errors.voucher_prefix && <p className="mt-1 text-sm text-red-600">{settingsForm.errors.voucher_prefix}</p>}
                                    </div>

                                    {/* Submit */}
                                    <div className="flex items-center justify-end">
                                        <button
                                            type="submit"
                                            disabled={settingsForm.processing}
                                            className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50"
                                        >
                                            {settingsForm.processing ? 'Saving...' : 'Save Settings'}
                                        </button>
                                    </div>
                                </form>
                            </div>
                        )}

                        {/* Categories Tab */}
                        {activeTab === 'categories' && (
                            <div className="p-6">
                                <div className="flex items-center justify-between mb-6">
                                    <div>
                                        <h3 className="text-lg font-medium text-gray-900">Cash Book Categories</h3>
                                        <p className="mt-1 text-sm text-gray-500">Manage income and expense categories for your cash book</p>
                                    </div>
                                    <button
                                        onClick={openCreateCategory}
                                        className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 transition-colors"
                                    >
                                        <svg className="w-4 h-4 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 4v16m8-8H4" /></svg>
                                        Add Category
                                    </button>
                                </div>

                                {/* Income Categories */}
                                <div className="mb-8">
                                    <h4 className="text-sm font-semibold text-gray-700 uppercase tracking-wider mb-3 flex items-center gap-2">
                                        <span className="inline-block w-3 h-3 rounded-full bg-green-500"></span>
                                        Income Categories ({incomeCategories.length})
                                    </h4>
                                    {incomeCategories.length === 0 ? (
                                        <div className="text-sm text-gray-500 border border-dashed border-gray-300 rounded-lg p-4 text-center">
                                            No income categories yet
                                        </div>
                                    ) : (
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                            {incomeCategories.map(category => (
                                                <div key={category.id} className="flex items-center justify-between bg-green-50 border border-green-200 rounded-lg px-4 py-3">
                                                    <span className="text-sm font-medium text-gray-900">{category.name}</span>
                                                    <div className="flex items-center gap-2">
                                                        <button
                                                            onClick={() => openEditCategory(category)}
                                                            className="text-blue-600 hover:text-blue-800 transition-colors"
                                                            title="Edit"
                                                        >
                                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" /></svg>
                                                        </button>
                                                        <button
                                                            onClick={() => setDeleteCategory(category)}
                                                            className="text-red-600 hover:text-red-800 transition-colors"
                                                            title="Delete"
                                                        >
                                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" /></svg>
                                                        </button>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    )}
                                </div>

                                {/* Expense Categories */}
                                <div>
                                    <h4 className="text-sm font-semibold text-gray-700 uppercase tracking-wider mb-3 flex items-center gap-2">
                                        <span className="inline-block w-3 h-3 rounded-full bg-red-500"></span>
                                        Expense Categories ({expenseCategories.length})
                                    </h4>
                                    {expenseCategories.length === 0 ? (
                                        <div className="text-sm text-gray-500 border border-dashed border-gray-300 rounded-lg p-4 text-center">
                                            No expense categories yet
                                        </div>
                                    ) : (
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                                            {expenseCategories.map(category => (
                                                <div key={category.id} className="flex items-center justify-between bg-red-50 border border-red-200 rounded-lg px-4 py-3">
                                                    <span className="text-sm font-medium text-gray-900">{category.name}</span>
                                                    <div className="flex items-center gap-2">
                                                        <button
                                                            onClick={() => openEditCategory(category)}
                                                            className="text-blue-600 hover:text-blue-800 transition-colors"
                                                            title="Edit"
                                                        >
                                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" /></svg>
                                                        </button>
                                                        <button
                                                            onClick={() => setDeleteCategory(category)}
                                                            className="text-red-600 hover:text-red-800 transition-colors"
                                                            title="Delete"
                                                        >
                                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" /></svg>
                                                        </button>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    )}
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>

            {/* Add/Edit Category Modal */}
            <FormModal
                show={showCategoryModal}
                onClose={() => setShowCategoryModal(false)}
                title={editingCategory ? 'Edit Category' : 'Add Category'}
                maxWidth="md"
            >
                <form onSubmit={handleCategorySubmit} className="space-y-5">
                    {/* Category Name */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-1">Category Name</label>
                        <input
                            type="text"
                            value={categoryForm.data.name}
                            onChange={e => categoryForm.setData('name', e.target.value)}
                            placeholder="e.g. Office Supplies, Salary, etc."
                            className="w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                            autoFocus
                        />
                        {categoryForm.errors.name && <p className="mt-1 text-sm text-red-600">{categoryForm.errors.name}</p>}
                    </div>

                    {/* Category Type */}
                    <div>
                        <label className="block text-sm font-medium text-gray-700 mb-2">Category Type</label>
                        <div className="grid grid-cols-2 gap-3">
                            <button
                                type="button"
                                onClick={() => categoryForm.setData('type', 'income')}
                                className={`flex items-center justify-center gap-2 px-4 py-3 rounded-lg border-2 transition-colors ${
                                    categoryForm.data.type === 'income'
                                        ? 'border-green-500 bg-green-50 text-green-700'
                                        : 'border-gray-200 bg-white text-gray-700 hover:border-gray-300'
                                }`}
                            >
                                <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M7 11l5-5m0 0l5 5m-5-5v12" /></svg>
                                <span className="font-medium">Cash In</span>
                            </button>
                            <button
                                type="button"
                                onClick={() => categoryForm.setData('type', 'expense')}
                                className={`flex items-center justify-center gap-2 px-4 py-3 rounded-lg border-2 transition-colors ${
                                    categoryForm.data.type === 'expense'
                                        ? 'border-red-500 bg-red-50 text-red-700'
                                        : 'border-gray-200 bg-white text-gray-700 hover:border-gray-300'
                                }`}
                            >
                                <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M17 13l-5 5m0 0l-5-5m5 5V6" /></svg>
                                <span className="font-medium">Cash Out</span>
                            </button>
                        </div>
                        {categoryForm.errors.type && <p className="mt-1 text-sm text-red-600">{categoryForm.errors.type}</p>}
                    </div>

                    {/* Actions */}
                    <div className="flex justify-end gap-3 pt-2 border-t border-gray-100">
                        <button
                            type="button"
                            onClick={() => setShowCategoryModal(false)}
                            className="rounded-md bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300 transition-colors"
                        >
                            Cancel
                        </button>
                        <button
                            type="submit"
                            disabled={categoryForm.processing}
                            className="rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50 transition-colors"
                        >
                            {categoryForm.processing ? 'Saving...' : editingCategory ? 'Update Category' : 'Add Category'}
                        </button>
                    </div>
                </form>
            </FormModal>

            {/* Delete Confirmation */}
            <ConfirmDeleteModal
                show={!!deleteCategory}
                onClose={() => setDeleteCategory(null)}
                onConfirm={handleDeleteCategory}
                title="Delete Category"
                message={`Are you sure you want to delete "${deleteCategory?.name}"? You can only delete categories with no existing entries.`}
            />
        </AuthenticatedLayout>
    );
}
