import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, Link, useForm } from '@inertiajs/react';
import { formatCurrency } from '@/utils';
import { useState, useEffect } from 'react';

export default function Create({ invoices, preselected_invoice_id }) {
    const { data, setData, post, processing, errors } = useForm({
        invoice_id: preselected_invoice_id || '',
        amount_paid: '',
        payment_date: new Date().toISOString().split('T')[0],
        payment_method: 'cash',
        proof_file: null,
        received_by: '',
        remarks: '',
    });

    const [selectedInvoice, setSelectedInvoice] = useState(null);

    useEffect(() => {
        if (data.invoice_id) {
            const inv = invoices.find((i) => i.id == data.invoice_id);
            setSelectedInvoice(inv || null);
        } else {
            setSelectedInvoice(null);
        }
    }, [data.invoice_id, invoices]);

    const getAlreadyPaid = (inv) => {
        return (inv?.payments || []).reduce((sum, p) => sum + parseFloat(p.amount_paid || 0), 0);
    };

    const alreadyPaid = selectedInvoice ? getAlreadyPaid(selectedInvoice) : 0;
    const totalAmount = selectedInvoice ? parseFloat(selectedInvoice.total_amount || 0) : 0;
    const remainingBalance = totalAmount - alreadyPaid;

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route('payments.store'), { forceFormData: true });
    };

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Record Payment</h2>}>
            <Head title="Record Payment" />

            <div className="py-12">
                <div className="mx-auto max-w-2xl sm:px-6 lg:px-8">
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium">Record New Payment</h3>
                                <Link href={route('payments.index')} className="inline-flex items-center rounded-md bg-gray-200 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-300">Back</Link>
                            </div>

                            <form onSubmit={handleSubmit} className="space-y-6">
                                {/* Invoice */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Invoice <span className="text-red-500">*</span></label>
                                    <select
                                        value={data.invoice_id}
                                        onChange={(e) => setData('invoice_id', e.target.value)}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    >
                                        <option value="">Select an invoice...</option>
                                        {invoices.map((inv) => {
                                            const paid = getAlreadyPaid(inv);
                                            const balance = parseFloat(inv.total_amount) - paid;
                                            return (
                                                <option key={inv.id} value={inv.id}>
                                                    {inv.invoice_number} - Shop {inv.lease?.shop?.shop_number} - {inv.lease?.tenant?.name} - Balance: {formatCurrency(balance)}
                                                </option>
                                            );
                                        })}
                                    </select>
                                    {errors.invoice_id && <p className="mt-1 text-sm text-red-600">{errors.invoice_id}</p>}
                                </div>

                                {/* Invoice Summary */}
                                {selectedInvoice && (
                                    <div className="rounded-lg border border-blue-200 bg-blue-50 p-4">
                                        <div className="grid grid-cols-3 gap-4 text-sm">
                                            <div>
                                                <span className="text-gray-500">Total Amount</span>
                                                <p className="font-semibold text-gray-900">{formatCurrency(totalAmount)}</p>
                                            </div>
                                            <div>
                                                <span className="text-gray-500">Already Paid</span>
                                                <p className="font-semibold text-green-700">{formatCurrency(alreadyPaid)}</p>
                                            </div>
                                            <div>
                                                <span className="text-gray-500">Remaining Balance</span>
                                                <p className="font-semibold text-red-700">{formatCurrency(remainingBalance)}</p>
                                            </div>
                                        </div>
                                    </div>
                                )}

                                {/* Amount Paid */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Amount Paid <span className="text-red-500">*</span></label>
                                    <input
                                        type="number"
                                        step="0.01"
                                        min="1"
                                        max={remainingBalance > 0 ? remainingBalance : undefined}
                                        value={data.amount_paid}
                                        onChange={(e) => setData('amount_paid', e.target.value)}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    />
                                    {remainingBalance > 0 && (
                                        <p className="mt-1 text-xs text-gray-500">Max: {formatCurrency(remainingBalance)}</p>
                                    )}
                                    {errors.amount_paid && <p className="mt-1 text-sm text-red-600">{errors.amount_paid}</p>}
                                </div>

                                {/* Payment Date */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Payment Date <span className="text-red-500">*</span></label>
                                    <input
                                        type="date"
                                        value={data.payment_date}
                                        onChange={(e) => setData('payment_date', e.target.value)}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    />
                                    {errors.payment_date && <p className="mt-1 text-sm text-red-600">{errors.payment_date}</p>}
                                </div>

                                {/* Payment Method */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Payment Method <span className="text-red-500">*</span></label>
                                    <select
                                        value={data.payment_method}
                                        onChange={(e) => setData('payment_method', e.target.value)}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    >
                                        <option value="cash">Cash</option>
                                        <option value="bank_transfer">Bank Transfer</option>
                                        <option value="online">Online</option>
                                    </select>
                                    {errors.payment_method && <p className="mt-1 text-sm text-red-600">{errors.payment_method}</p>}
                                </div>

                                {/* Proof File */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Proof File</label>
                                    <input
                                        type="file"
                                        onChange={(e) => setData('proof_file', e.target.files[0])}
                                        className="mt-1 block w-full text-sm text-gray-500 file:mr-4 file:rounded-md file:border-0 file:bg-blue-50 file:px-4 file:py-2 file:text-sm file:font-medium file:text-blue-700 hover:file:bg-blue-100"
                                        accept="image/*,.pdf"
                                    />
                                    {data.payment_method === 'bank_transfer' && (
                                        <p className="mt-1 text-xs text-amber-600">Required for bank transfer payments</p>
                                    )}
                                    {errors.proof_file && <p className="mt-1 text-sm text-red-600">{errors.proof_file}</p>}
                                </div>

                                {/* Received By */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Received By</label>
                                    <input
                                        type="text"
                                        value={data.received_by}
                                        onChange={(e) => setData('received_by', e.target.value)}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    />
                                    {errors.received_by && <p className="mt-1 text-sm text-red-600">{errors.received_by}</p>}
                                </div>

                                {/* Remarks */}
                                <div>
                                    <label className="block text-sm font-medium text-gray-700">Remarks</label>
                                    <textarea
                                        value={data.remarks}
                                        onChange={(e) => setData('remarks', e.target.value)}
                                        rows={3}
                                        className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    />
                                    {errors.remarks && <p className="mt-1 text-sm text-red-600">{errors.remarks}</p>}
                                </div>

                                {/* Submit */}
                                <div className="flex items-center justify-end gap-4">
                                    <Link href={route('payments.index')} className="text-sm text-gray-600 hover:text-gray-900">Cancel</Link>
                                    <button
                                        type="submit"
                                        disabled={processing}
                                        className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50"
                                    >
                                        {processing ? 'Saving...' : 'Record Payment'}
                                    </button>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
}
