import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, Link } from '@inertiajs/react';
import { formatCurrency, formatDate } from '@/utils';

export default function Show({ owner }) {
    const shops = owner.shops || [];
    const ownerPayments = owner.owner_payments || [];

    return (
        <AuthenticatedLayout
            header={
                <div className="flex items-center justify-between">
                    <h2 className="text-xl font-semibold leading-tight text-gray-800">
                        Owner – {owner.name}
                    </h2>
                    <Link
                        href={route('owners.edit', owner.id)}
                        className="inline-flex items-center rounded-md bg-indigo-600 px-4 py-2 text-sm font-medium text-white shadow-sm hover:bg-indigo-700"
                    >
                        Edit Owner
                    </Link>
                </div>
            }
        >
            <Head title={`Owner – ${owner.name}`} />

            <div className="py-6">
                <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
                    {/* Owner Details Card */}
                    <div className="rounded-lg bg-white p-6 shadow">
                        <h3 className="mb-4 text-lg font-semibold text-gray-800">Owner Details</h3>
                        <div className="grid grid-cols-1 gap-4 sm:grid-cols-2 lg:grid-cols-3">
                            <div>
                                <span className="text-sm font-medium text-gray-500">Name</span>
                                <p className="text-gray-900">{owner.name}</p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">Phone</span>
                                <p className="text-gray-900">{owner.phone || '-'}</p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">Email</span>
                                <p className="text-gray-900">{owner.email || '-'}</p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">CNIC</span>
                                <p className="text-gray-900">{owner.cnic || '-'}</p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">Address</span>
                                <p className="text-gray-900">{owner.address || '-'}</p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">Bank</span>
                                <p className="text-gray-900">
                                    {owner.bank_name
                                        ? `${owner.bank_name} – ${owner.bank_account_no || 'N/A'}`
                                        : '-'}
                                </p>
                            </div>
                            {owner.notes && (
                                <div className="sm:col-span-2 lg:col-span-3">
                                    <span className="text-sm font-medium text-gray-500">Notes</span>
                                    <p className="whitespace-pre-line text-gray-900">{owner.notes}</p>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Shops */}
                    <div className="mt-8">
                        <h3 className="mb-4 text-lg font-semibold text-gray-800">
                            Shops ({shops.length})
                        </h3>
                        <div className="overflow-hidden rounded-lg bg-white shadow">
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Shop #</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Status</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Current Tenant</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium uppercase text-gray-500">Default Rent</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200">
                                        {shops.length === 0 && (
                                            <tr>
                                                <td colSpan="4" className="px-4 py-6 text-center text-gray-400">
                                                    No shops assigned to this owner.
                                                </td>
                                            </tr>
                                        )}
                                        {shops.map((shop) => {
                                            const activeLease = shop.active_lease;
                                            const isOccupied = !!activeLease;
                                            return (
                                                <tr key={shop.id} className="hover:bg-gray-50">
                                                    <td className="whitespace-nowrap px-4 py-3 text-sm font-medium text-gray-900">
                                                        <Link
                                                            href={route('shops.show', shop.id)}
                                                            className="text-blue-600 hover:text-blue-800"
                                                        >
                                                            {shop.shop_number}
                                                        </Link>
                                                    </td>
                                                    <td className="whitespace-nowrap px-4 py-3 text-sm">
                                                        <span
                                                            className={`inline-flex rounded-full px-2 py-0.5 text-xs font-semibold ${
                                                                isOccupied
                                                                    ? 'bg-green-100 text-green-800'
                                                                    : 'bg-gray-100 text-gray-600'
                                                            }`}
                                                        >
                                                            {isOccupied ? 'Occupied' : 'Vacant'}
                                                        </span>
                                                    </td>
                                                    <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700">
                                                        {activeLease?.tenant?.name || '-'}
                                                    </td>
                                                    <td className="whitespace-nowrap px-4 py-3 text-right text-sm text-gray-700">
                                                        {formatCurrency(shop.default_rent)}
                                                    </td>
                                                </tr>
                                            );
                                        })}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    {/* Owner Payments */}
                    <div className="mt-8">
                        <h3 className="mb-4 text-lg font-semibold text-gray-800">
                            Owner Payments ({ownerPayments.length})
                        </h3>
                        <div className="overflow-hidden rounded-lg bg-white shadow">
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Voucher #</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Date</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium uppercase text-gray-500">Amount</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase text-gray-500">Method</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200">
                                        {ownerPayments.length === 0 && (
                                            <tr>
                                                <td colSpan="4" className="px-4 py-6 text-center text-gray-400">
                                                    No payments recorded yet.
                                                </td>
                                            </tr>
                                        )}
                                        {ownerPayments.map((payment) => (
                                            <tr key={payment.id} className="hover:bg-gray-50">
                                                <td className="whitespace-nowrap px-4 py-3 text-sm font-medium text-gray-900">
                                                    {payment.voucher_number || '-'}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-700">
                                                    {formatDate(payment.payment_date)}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-right text-sm font-medium text-gray-900">
                                                    {formatCurrency(payment.amount)}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm capitalize text-gray-700">
                                                    {payment.payment_method || '-'}
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>

                    {/* Back Link */}
                    <div className="mt-6">
                        <Link
                            href={route('owners.index')}
                            className="text-sm text-blue-600 hover:text-blue-800"
                        >
                            ← Back to Owners
                        </Link>
                    </div>
                </div>
            </div>
        </AuthenticatedLayout>
    );
}
