import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import ConfirmDeleteModal from '@/Components/ConfirmDeleteModal';
import { Head, Link, router, usePage } from '@inertiajs/react';
import { formatCurrency, formatDate } from '@/utils';
import { useState } from 'react';

export default function Index({ payments, owners, ownerSummaries, grandTotals, filters }) {
    const { auth } = usePage().props;
    const [deleteId, setDeleteId] = useState(null);
    const [deleting, setDeleting] = useState(false);
    const [selectedOwnerIds, setSelectedOwnerIds] = useState(
        filters.owner_ids ? filters.owner_ids.split(',').filter(Boolean) : []
    );
    const [showAllOwners, setShowAllOwners] = useState(false);

    const toggleOwner = (id) => {
        const idStr = String(id);
        const newIds = selectedOwnerIds.includes(idStr)
            ? selectedOwnerIds.filter((i) => i !== idStr)
            : [...selectedOwnerIds, idStr];
        setSelectedOwnerIds(newIds);
        router.get(route('owner-payments.index'), { owner_ids: newIds.join(',') }, { preserveState: true, replace: true });
    };

    const clearFilters = () => {
        setSelectedOwnerIds([]);
        router.get(route('owner-payments.index'), {}, { preserveState: true, replace: true });
    };

    const handleDelete = () => {
        setDeleting(true);
        router.delete(route('owner-payments.destroy', deleteId), {
            onFinish: () => { setDeleting(false); setDeleteId(null); },
        });
    };

    // Filter summaries based on selection
    const filteredSummaries = selectedOwnerIds.length > 0
        ? ownerSummaries.filter((s) => selectedOwnerIds.includes(String(s.id)))
        : ownerSummaries;

    const displayedSummaries = showAllOwners ? filteredSummaries : filteredSummaries.slice(0, 10);

    // Recalculate grand totals for filtered view
    const filteredTotals = selectedOwnerIds.length > 0
        ? {
            collected: filteredSummaries.reduce((s, o) => s + o.total_collected, 0),
            paid: filteredSummaries.reduce((s, o) => s + o.total_paid, 0),
            pending: filteredSummaries.reduce((s, o) => s + o.pending, 0),
        }
        : grandTotals;

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Owner Payments</h2>}>
            <Head title="Owner Payments" />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-8 space-y-6">

                    {/* Grand Summary Cards */}
                    <div className="grid grid-cols-1 gap-4 sm:grid-cols-3">
                        <div className="rounded-xl bg-gradient-to-br from-blue-500 to-blue-600 p-5 text-white shadow-lg">
                            <div className="flex items-center gap-3">
                                <div className="flex h-11 w-11 items-center justify-center rounded-lg bg-white/20">
                                    <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M2.25 18.75a60.07 60.07 0 0 1 15.797 2.101c.727.198 1.453-.342 1.453-1.096V18.75M3.75 4.5v.75A.75.75 0 0 1 3 6h-.75m0 0v-.375c0-.621.504-1.125 1.125-1.125H20.25M2.25 6v9m18-10.5v.75c0 .414.336.75.75.75h.75m-1.5-1.5h.375c.621 0 1.125.504 1.125 1.125v9.75c0 .621-.504 1.125-1.125 1.125h-.375m1.5-1.5H21a.75.75 0 0 0-.75.75v.75m0 0H3.75m0 0h-.375a1.125 1.125 0 0 1-1.125-1.125V15m1.5 1.5v-.75A.75.75 0 0 0 3 15h-.75M15 10.5a3 3 0 1 1-6 0 3 3 0 0 1 6 0Zm3 0h.008v.008H18V10.5Zm-12 0h.008v.008H6V10.5Z" /></svg>
                                </div>
                                <div>
                                    <p className="text-sm font-medium text-blue-100">Total Collected</p>
                                    <p className="text-2xl font-bold">{formatCurrency(filteredTotals.collected)}</p>
                                </div>
                            </div>
                            <p className="mt-2 text-xs text-blue-200">From tenant invoice payments</p>
                        </div>
                        <div className="rounded-xl bg-gradient-to-br from-green-500 to-green-600 p-5 text-white shadow-lg">
                            <div className="flex items-center gap-3">
                                <div className="flex h-11 w-11 items-center justify-center rounded-lg bg-white/20">
                                    <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M9 12.75 11.25 15 15 9.75M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z" /></svg>
                                </div>
                                <div>
                                    <p className="text-sm font-medium text-green-100">Paid to Owners</p>
                                    <p className="text-2xl font-bold">{formatCurrency(filteredTotals.paid)}</p>
                                </div>
                            </div>
                            <p className="mt-2 text-xs text-green-200">Total disbursed to shop owners</p>
                        </div>
                        <div className="rounded-xl bg-gradient-to-br from-amber-500 to-orange-500 p-5 text-white shadow-lg">
                            <div className="flex items-center gap-3">
                                <div className="flex h-11 w-11 items-center justify-center rounded-lg bg-white/20">
                                    <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M12 6v6h4.5m4.5 0a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z" /></svg>
                                </div>
                                <div>
                                    <p className="text-sm font-medium text-amber-100">Pending</p>
                                    <p className="text-2xl font-bold">{formatCurrency(filteredTotals.pending)}</p>
                                </div>
                            </div>
                            <p className="mt-2 text-xs text-amber-200">Yet to be paid to owners</p>
                        </div>
                    </div>

                    {/* Owner-wise Summary Table */}
                    <div className="bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
                                    <svg className="h-5 w-5 text-blue-600" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M15 19.128a9.38 9.38 0 0 0 2.625.372 9.337 9.337 0 0 0 4.121-.952 4.125 4.125 0 0 0-7.533-2.493M15 19.128v-.003c0-1.113-.285-2.16-.786-3.07M15 19.128v.106A12.318 12.318 0 0 1 8.624 21c-2.331 0-4.512-.645-6.374-1.766l-.001-.109a6.375 6.375 0 0 1 11.964-3.07M12 6.375a3.375 3.375 0 1 1-6.75 0 3.375 3.375 0 0 1 6.75 0Zm8.25 2.25a2.625 2.625 0 1 1-5.25 0 2.625 2.625 0 0 1 5.25 0Z" /></svg>
                                    Owner-wise Summary
                                </h3>
                                <div className="flex items-center gap-2">
                                    {selectedOwnerIds.length > 0 && (
                                        <button onClick={clearFilters} className="text-xs text-blue-600 hover:text-blue-800 underline">
                                            Clear filters ({selectedOwnerIds.length})
                                        </button>
                                    )}
                                    <Link
                                        href={route('owner-payments.create')}
                                        className="inline-flex items-center gap-1.5 rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 transition-colors"
                                    >
                                        <svg className="h-4 w-4" fill="none" viewBox="0 0 24 24" strokeWidth="2" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M12 4.5v15m7.5-7.5h-15" /></svg>
                                        Pay Owner
                                    </Link>
                                </div>
                            </div>

                            {/* Owner Filter Chips */}
                            <div className="mb-4 flex flex-wrap gap-2">
                                {owners.map((owner) => {
                                    const isSelected = selectedOwnerIds.includes(String(owner.id));
                                    const summary = ownerSummaries.find((s) => s.id === owner.id);
                                    const hasPending = summary && summary.pending > 0;
                                    return (
                                        <button
                                            key={owner.id}
                                            onClick={() => toggleOwner(owner.id)}
                                            className={`inline-flex items-center gap-1.5 rounded-full px-3 py-1.5 text-xs font-medium transition-all ${
                                                isSelected
                                                    ? 'bg-blue-600 text-white shadow-sm'
                                                    : hasPending
                                                    ? 'bg-amber-50 text-amber-700 border border-amber-200 hover:bg-amber-100'
                                                    : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
                                            }`}
                                        >
                                            {owner.name}
                                            {hasPending && !isSelected && (
                                                <span className="inline-block h-1.5 w-1.5 rounded-full bg-amber-500" />
                                            )}
                                        </button>
                                    );
                                })}
                            </div>

                            {/* Summary Table */}
                            <div className="overflow-x-auto rounded-lg border border-gray-200">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-semibold uppercase tracking-wider text-gray-500">Owner</th>
                                            <th className="px-4 py-3 text-center text-xs font-semibold uppercase tracking-wider text-gray-500">Shops</th>
                                            <th className="px-4 py-3 text-right text-xs font-semibold uppercase tracking-wider text-gray-500">Collected</th>
                                            <th className="px-4 py-3 text-right text-xs font-semibold uppercase tracking-wider text-gray-500">Paid</th>
                                            <th className="px-4 py-3 text-right text-xs font-semibold uppercase tracking-wider text-gray-500">Pending</th>
                                            <th className="px-4 py-3 text-center text-xs font-semibold uppercase tracking-wider text-gray-500">Status</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 bg-white">
                                        {displayedSummaries.map((s) => (
                                            <tr key={s.id} className="hover:bg-gray-50 cursor-pointer" onClick={() => toggleOwner(s.id)}>
                                                <td className="whitespace-nowrap px-4 py-3">
                                                    <div className="flex items-center gap-2">
                                                        <div className="flex h-8 w-8 items-center justify-center rounded-full bg-blue-100 text-xs font-bold text-blue-700">
                                                            {s.name?.charAt(0)?.toUpperCase()}
                                                        </div>
                                                        <div>
                                                            <p className="text-sm font-medium text-gray-900">{s.name}</p>
                                                            {s.phone && <p className="text-xs text-gray-400">{s.phone}</p>}
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-center">
                                                    <span className="inline-flex items-center rounded-full bg-gray-100 px-2 py-0.5 text-xs font-medium text-gray-700">
                                                        {s.shops_count} shop{s.shops_count !== 1 ? 's' : ''}
                                                    </span>
                                                    <p className="text-[10px] text-gray-400 mt-0.5">{s.shop_numbers.join(', ')}</p>
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-right text-sm font-medium text-blue-700">{formatCurrency(s.total_collected)}</td>
                                                <td className="whitespace-nowrap px-4 py-3 text-right text-sm font-medium text-green-700">{formatCurrency(s.total_paid)}</td>
                                                <td className="whitespace-nowrap px-4 py-3 text-right text-sm font-bold text-amber-700">{formatCurrency(s.pending)}</td>
                                                <td className="whitespace-nowrap px-4 py-3 text-center">
                                                    {s.pending <= 0 ? (
                                                        <span className="inline-flex items-center rounded-full bg-green-100 px-2 py-0.5 text-xs font-semibold text-green-800">All Clear</span>
                                                    ) : (
                                                        <span className="inline-flex items-center rounded-full bg-amber-100 px-2 py-0.5 text-xs font-semibold text-amber-800">Pending</span>
                                                    )}
                                                </td>
                                            </tr>
                                        ))}
                                        {displayedSummaries.length === 0 && (
                                            <tr>
                                                <td colSpan="6" className="px-4 py-8 text-center text-sm text-gray-400">No owners found.</td>
                                            </tr>
                                        )}
                                    </tbody>
                                    {filteredSummaries.length > 0 && (
                                        <tfoot className="bg-gray-50">
                                            <tr className="font-bold">
                                                <td className="px-4 py-3 text-sm text-gray-700" colSpan="2">Grand Total</td>
                                                <td className="px-4 py-3 text-right text-sm text-blue-800">{formatCurrency(filteredTotals.collected)}</td>
                                                <td className="px-4 py-3 text-right text-sm text-green-800">{formatCurrency(filteredTotals.paid)}</td>
                                                <td className="px-4 py-3 text-right text-sm text-amber-800">{formatCurrency(filteredTotals.pending)}</td>
                                                <td></td>
                                            </tr>
                                        </tfoot>
                                    )}
                                </table>
                            </div>

                            {filteredSummaries.length > 10 && !showAllOwners && (
                                <button onClick={() => setShowAllOwners(true)} className="mt-3 text-sm text-blue-600 hover:text-blue-800">
                                    Show all {filteredSummaries.length} owners...
                                </button>
                            )}
                        </div>
                    </div>

                    {/* Payment Vouchers List */}
                    <div className="bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex items-center justify-between mb-4">
                                <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
                                    <svg className="h-5 w-5 text-green-600" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M9 12h3.75M9 15h3.75M9 18h3.75m3 .75H18a2.25 2.25 0 0 0 2.25-2.25V6.108c0-1.135-.845-2.098-1.976-2.192a48.424 48.424 0 0 0-1.123-.08m-5.801 0c-.065.21-.1.433-.1.664 0 .414.336.75.75.75h4.5a.75.75 0 0 0 .75-.75 2.25 2.25 0 0 0-.1-.664m-5.8 0A2.251 2.251 0 0 1 13.5 2.25H15c1.012 0 1.867.668 2.15 1.586m-5.8 0c-.376.023-.75.05-1.124.08C9.095 4.01 8.25 4.973 8.25 6.108V8.25m0 0H4.875c-.621 0-1.125.504-1.125 1.125v11.25c0 .621.504 1.125 1.125 1.125h9.75c.621 0 1.125-.504 1.125-1.125V9.375c0-.621-.504-1.125-1.125-1.125H8.25ZM6.75 12h.008v.008H6.75V12Zm0 3h.008v.008H6.75V15Zm0 3h.008v.008H6.75V18Z" /></svg>
                                    Payment Vouchers
                                </h3>
                            </div>

                            <div className="overflow-x-auto rounded-lg border border-gray-200">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-semibold uppercase tracking-wider text-gray-500">Voucher#</th>
                                            <th className="px-4 py-3 text-left text-xs font-semibold uppercase tracking-wider text-gray-500">Owner</th>
                                            <th className="px-4 py-3 text-left text-xs font-semibold uppercase tracking-wider text-gray-500">Date</th>
                                            <th className="px-4 py-3 text-right text-xs font-semibold uppercase tracking-wider text-gray-500">Amount</th>
                                            <th className="px-4 py-3 text-center text-xs font-semibold uppercase tracking-wider text-gray-500">Method</th>
                                            <th className="px-4 py-3 text-center text-xs font-semibold uppercase tracking-wider text-gray-500">Shops</th>
                                            <th className="px-4 py-3 text-center text-xs font-semibold uppercase tracking-wider text-gray-500">Signature</th>
                                            <th className="px-4 py-3 text-center text-xs font-semibold uppercase tracking-wider text-gray-500">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 bg-white">
                                        {payments.data.map((payment) => (
                                            <tr key={payment.id} className="hover:bg-gray-50">
                                                <td className="whitespace-nowrap px-4 py-3 text-sm font-semibold text-gray-900">{payment.voucher_number}</td>
                                                <td className="whitespace-nowrap px-4 py-3">
                                                    <p className="text-sm font-medium text-gray-900">{payment.owner?.name || '—'}</p>
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-gray-500">{formatDate(payment.payment_date)}</td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-right font-bold text-green-700">{formatCurrency(payment.total_amount)}</td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-center">
                                                    <span className={`inline-flex items-center gap-1 rounded-full px-2 py-0.5 text-xs font-medium ${payment.payment_method === 'cash' ? 'bg-amber-100 text-amber-800' : 'bg-blue-100 text-blue-800'}`}>
                                                        {payment.payment_method === 'cash' ? '💵' : '🏦'} {payment.payment_method?.replace('_', ' ')}
                                                    </span>
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-xs text-center text-gray-500">
                                                    {payment.items?.map((i) => i.shop?.shop_number).filter(Boolean).join(', ') || '—'}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-center">
                                                    {payment.owner_signature_taken ? (
                                                        <span className="inline-flex items-center rounded-full bg-green-100 px-2 py-0.5 text-xs font-medium text-green-700">Signed ✓</span>
                                                    ) : (
                                                        <span className="inline-flex items-center rounded-full bg-gray-100 px-2 py-0.5 text-xs font-medium text-gray-500">Not Signed</span>
                                                    )}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-3 text-sm text-center">
                                                    <div className="flex items-center justify-center gap-1">
                                                        <Link href={route('owner-payments.show', payment.id)} className="rounded-md p-1.5 text-gray-400 hover:bg-blue-50 hover:text-blue-600" title="View">
                                                            <svg className="h-4 w-4" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M2.036 12.322a1.012 1.012 0 0 1 0-.639C3.423 7.51 7.36 4.5 12 4.5c4.638 0 8.573 3.007 9.963 7.178.07.207.07.431 0 .639C20.577 16.49 16.64 19.5 12 19.5c-4.638 0-8.573-3.007-9.963-7.178Z" /><path strokeLinecap="round" strokeLinejoin="round" d="M15 12a3 3 0 1 1-6 0 3 3 0 0 1 6 0Z" /></svg>
                                                        </Link>
                                                        <Link href={route('owner-payments.print', payment.id)} target="_blank" className="rounded-md p-1.5 text-gray-400 hover:bg-green-50 hover:text-green-600" title="Print">
                                                            <svg className="h-4 w-4" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M6.72 13.829c-.24.03-.48.062-.72.096m.72-.096a42.415 42.415 0 0 1 10.56 0m-10.56 0L6.34 18m10.94-4.171c.24.03.48.062.72.096m-.72-.096L17.66 18m0 0 .229 2.523a1.125 1.125 0 0 1-1.12 1.227H7.231c-.662 0-1.18-.568-1.12-1.227L6.34 18m11.318 0h1.091A2.25 2.25 0 0 0 21 15.75V9.456c0-1.081-.768-2.015-1.837-2.175a48.055 48.055 0 0 0-1.913-.247M6.34 18H5.25A2.25 2.25 0 0 1 3 15.75V9.456c0-1.081.768-2.015 1.837-2.175a48.041 48.041 0 0 1 1.913-.247m10.5 0a48.536 48.536 0 0 0-10.5 0m10.5 0V3.375c0-.621-.504-1.125-1.125-1.125h-8.25c-.621 0-1.125.504-1.125 1.125v3.659M18.75 7.131a48.041 48.041 0 0 0-1.913-.247M6.75 7.131a48.041 48.041 0 0 1-1.913-.247" /></svg>
                                                        </Link>
                                                        {auth.user.is_admin && (
                                                            <button onClick={() => setDeleteId(payment.id)} className="rounded-md p-1.5 text-gray-400 hover:bg-red-50 hover:text-red-600" title="Delete">
                                                                <svg className="h-4 w-4" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="m14.74 9-.346 9m-4.788 0L9.26 9m9.968-3.21c.342.052.682.107 1.022.166m-1.022-.165L18.16 19.673a2.25 2.25 0 0 1-2.244 2.077H8.084a2.25 2.25 0 0 1-2.244-2.077L4.772 5.79m14.456 0a48.108 48.108 0 0 0-3.478-.397m-12 .562c.34-.059.68-.114 1.022-.165m0 0a48.11 48.11 0 0 1 3.478-.397m7.5 0v-.916c0-1.18-.91-2.164-2.09-2.201a51.964 51.964 0 0 0-3.32 0c-1.18.037-2.09 1.022-2.09 2.201v.916m7.5 0a48.667 48.667 0 0 0-7.5 0" /></svg>
                                                            </button>
                                                        )}
                                                    </div>
                                                </td>
                                            </tr>
                                        ))}
                                        {payments.data.length === 0 && (
                                            <tr>
                                                <td colSpan="8" className="px-4 py-8 text-center text-sm text-gray-400">No payment vouchers found.</td>
                                            </tr>
                                        )}
                                    </tbody>
                                </table>
                            </div>

                            {/* Pagination */}
                            {payments.links && payments.links.length > 3 && (
                                <div className="flex flex-wrap items-center gap-1 mt-4">
                                    {payments.links.map((link, i) => (
                                        link.url ? (
                                            <Link key={i} href={link.url} className={`px-3 py-1 text-sm rounded ${link.active ? 'bg-blue-600 text-white' : 'bg-white border hover:bg-gray-50'}`}
                                                dangerouslySetInnerHTML={{ __html: link.label }} preserveState />
                                        ) : (
                                            <span key={i} className="px-3 py-1 text-sm text-gray-400" dangerouslySetInnerHTML={{ __html: link.label }} />
                                        )
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>

            <ConfirmDeleteModal
                show={deleteId !== null}
                onClose={() => setDeleteId(null)}
                onConfirm={handleDelete}
                title="Delete Owner Payment"
                message="Are you sure you want to delete this payment voucher? This action cannot be undone."
                processing={deleting}
            />
        </AuthenticatedLayout>
    );
}
