import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, Link, useForm } from '@inertiajs/react';
import { formatCurrency } from '@/utils';
import { useState, useEffect } from 'react';

export default function Create({ owners }) {
    const currentMonth = new Date().toISOString().slice(0, 7);

    const { data, setData, post, processing, errors } = useForm({
        owner_id: '',
        payment_date: new Date().toISOString().split('T')[0],
        payment_method: 'cash',
        proof_file: null,
        owner_signature_taken: false,
        remarks: '',
        items: [],
    });

    const [selectedOwner, setSelectedOwner] = useState(null);
    const [shopSelections, setShopSelections] = useState({});

    useEffect(() => {
        if (data.owner_id) {
            const owner = owners.find((o) => o.id == data.owner_id);
            setSelectedOwner(owner || null);
            if (owner) {
                const selections = {};
                owner.shops.forEach((shop) => {
                    selections[shop.id] = {
                        checked: false,
                        amount: shop.pending > 0 ? shop.pending : (shop.agreed_rent || 0),
                        month: currentMonth,
                    };
                });
                setShopSelections(selections);
            } else {
                setShopSelections({});
            }
        } else {
            setSelectedOwner(null);
            setShopSelections({});
        }
    }, [data.owner_id]);

    useEffect(() => {
        const items = [];
        Object.entries(shopSelections).forEach(([shopId, sel]) => {
            if (sel.checked) {
                items.push({
                    shop_id: parseInt(shopId),
                    month: sel.month || currentMonth,
                    amount: parseFloat(sel.amount) || 0,
                });
            }
        });
        setData('items', items);
    }, [shopSelections]);

    const toggleShop = (shopId) => {
        setShopSelections((prev) => ({
            ...prev,
            [shopId]: { ...prev[shopId], checked: !prev[shopId].checked },
        }));
    };

    const selectAllShops = () => {
        setShopSelections((prev) => {
            const next = { ...prev };
            const allChecked = Object.values(next).every((s) => s.checked);
            Object.keys(next).forEach((id) => { next[id] = { ...next[id], checked: !allChecked }; });
            return next;
        });
    };

    const updateShopField = (shopId, field, value) => {
        setShopSelections((prev) => ({
            ...prev,
            [shopId]: { ...prev[shopId], [field]: value },
        }));
    };

    const totalAmount = data.items.reduce((sum, item) => sum + (parseFloat(item.amount) || 0), 0);

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route('owner-payments.store'), { forceFormData: true });
    };

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Pay Owner</h2>}>
            <Head title="Pay Owner" />

            <div className="py-12">
                <div className="mx-auto max-w-4xl sm:px-6 lg:px-8 space-y-6">

                    {/* Back Button */}
                    <div>
                        <Link href={route('owner-payments.index')} className="inline-flex items-center gap-1.5 text-sm text-gray-600 hover:text-gray-900">
                            <svg className="h-4 w-4" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M10.5 19.5 3 12m0 0 7.5-7.5M3 12h18" /></svg>
                            Back to Owner Payments
                        </Link>
                    </div>

                    <form onSubmit={handleSubmit} className="space-y-6">

                        {/* Step 1: Select Owner */}
                        <div className="bg-white shadow-sm sm:rounded-lg p-6">
                            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                                <span className="flex h-7 w-7 items-center justify-center rounded-full bg-blue-100 text-xs font-bold text-blue-700">1</span>
                                Select Owner
                            </h3>
                            <select
                                value={data.owner_id}
                                onChange={(e) => setData('owner_id', e.target.value)}
                                className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                            >
                                <option value="">— Choose an owner —</option>
                                {owners.map((owner) => (
                                    <option key={owner.id} value={owner.id}>
                                        {owner.name} — {owner.shops.length} shop(s) — Pending: {formatCurrency(owner.pending)}
                                    </option>
                                ))}
                            </select>
                            {errors.owner_id && <p className="mt-1 text-sm text-red-600">{errors.owner_id}</p>}

                            {/* Owner Summary Card */}
                            {selectedOwner && (
                                <div className="mt-4 rounded-xl bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-100 p-4">
                                    <div className="flex items-center gap-3 mb-3">
                                        <div className="flex h-10 w-10 items-center justify-center rounded-full bg-blue-600 text-white font-bold">
                                            {selectedOwner.name?.charAt(0)?.toUpperCase()}
                                        </div>
                                        <div>
                                            <p className="font-semibold text-gray-900">{selectedOwner.name}</p>
                                            <p className="text-xs text-gray-500">
                                                {selectedOwner.phone && `Phone: ${selectedOwner.phone}`}
                                                {selectedOwner.bank_name && ` • Bank: ${selectedOwner.bank_name}`}
                                                {selectedOwner.bank_account_no && ` (${selectedOwner.bank_account_no})`}
                                            </p>
                                        </div>
                                    </div>
                                    <div className="grid grid-cols-3 gap-3">
                                        <div className="rounded-lg bg-white p-3 text-center border border-blue-100">
                                            <p className="text-[10px] font-medium text-gray-500 uppercase">Collected</p>
                                            <p className="text-sm font-bold text-blue-700">{formatCurrency(selectedOwner.total_collected)}</p>
                                        </div>
                                        <div className="rounded-lg bg-white p-3 text-center border border-green-100">
                                            <p className="text-[10px] font-medium text-gray-500 uppercase">Paid</p>
                                            <p className="text-sm font-bold text-green-700">{formatCurrency(selectedOwner.total_paid)}</p>
                                        </div>
                                        <div className="rounded-lg bg-white p-3 text-center border border-amber-100">
                                            <p className="text-[10px] font-medium text-gray-500 uppercase">Pending</p>
                                            <p className="text-sm font-bold text-amber-700">{formatCurrency(selectedOwner.pending)}</p>
                                        </div>
                                    </div>
                                </div>
                            )}
                        </div>

                        {/* Step 2: Select Shops */}
                        {selectedOwner && selectedOwner.shops.length > 0 && (
                            <div className="bg-white shadow-sm sm:rounded-lg p-6">
                                <div className="flex items-center justify-between mb-4">
                                    <h3 className="text-lg font-semibold text-gray-900 flex items-center gap-2">
                                        <span className="flex h-7 w-7 items-center justify-center rounded-full bg-blue-100 text-xs font-bold text-blue-700">2</span>
                                        Select Shops & Amounts
                                    </h3>
                                    <button type="button" onClick={selectAllShops} className="text-xs text-blue-600 hover:text-blue-800 underline">
                                        {Object.values(shopSelections).every((s) => s.checked) ? 'Deselect All' : 'Select All'}
                                    </button>
                                </div>

                                <div className="space-y-3">
                                    {selectedOwner.shops.map((shop) => {
                                        const sel = shopSelections[shop.id] || { checked: false, amount: 0, month: currentMonth };
                                        return (
                                            <div
                                                key={shop.id}
                                                className={`rounded-xl border-2 p-4 transition-all ${
                                                    sel.checked
                                                        ? 'border-blue-400 bg-blue-50/50 shadow-sm'
                                                        : 'border-gray-200 bg-white hover:border-gray-300'
                                                }`}
                                            >
                                                <div className="flex items-start gap-3">
                                                    <input
                                                        type="checkbox"
                                                        checked={sel.checked}
                                                        onChange={() => toggleShop(shop.id)}
                                                        className="mt-1 h-4 w-4 rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                                    />
                                                    <div className="flex-1 min-w-0">
                                                        <div className="flex items-center justify-between">
                                                            <div>
                                                                <p className="font-semibold text-gray-900">Shop #{shop.shop_number}</p>
                                                                {shop.has_lease ? (
                                                                    <p className="text-xs text-gray-500">Rent: {formatCurrency(shop.agreed_rent)}</p>
                                                                ) : (
                                                                    <p className="text-xs text-yellow-600">No active lease</p>
                                                                )}
                                                            </div>
                                                            <div className="text-right">
                                                                <div className="grid grid-cols-3 gap-3 text-center">
                                                                    <div>
                                                                        <p className="text-[9px] font-medium text-gray-400 uppercase">Collected</p>
                                                                        <p className="text-xs font-bold text-blue-600">{formatCurrency(shop.collected)}</p>
                                                                    </div>
                                                                    <div>
                                                                        <p className="text-[9px] font-medium text-gray-400 uppercase">Paid</p>
                                                                        <p className="text-xs font-bold text-green-600">{formatCurrency(shop.paid)}</p>
                                                                    </div>
                                                                    <div>
                                                                        <p className="text-[9px] font-medium text-gray-400 uppercase">Pending</p>
                                                                        <p className={`text-xs font-bold ${shop.pending > 0 ? 'text-amber-600' : 'text-green-600'}`}>{formatCurrency(shop.pending)}</p>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                        </div>

                                                        {sel.checked && (
                                                            <div className="mt-3 grid grid-cols-2 gap-3">
                                                                <div>
                                                                    <label className="block text-xs font-medium text-gray-600 mb-1">Amount</label>
                                                                    <input
                                                                        type="number"
                                                                        step="0.01"
                                                                        min="0"
                                                                        value={sel.amount}
                                                                        onChange={(e) => updateShopField(shop.id, 'amount', e.target.value)}
                                                                        className="w-full rounded-lg border-gray-300 text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                                                    />
                                                                </div>
                                                                <div>
                                                                    <label className="block text-xs font-medium text-gray-600 mb-1">For Month</label>
                                                                    <input
                                                                        type="month"
                                                                        value={sel.month}
                                                                        onChange={(e) => updateShopField(shop.id, 'month', e.target.value)}
                                                                        className="w-full rounded-lg border-gray-300 text-sm shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                                                    />
                                                                </div>
                                                            </div>
                                                        )}
                                                    </div>
                                                </div>
                                            </div>
                                        );
                                    })}
                                </div>
                                {errors.items && <p className="mt-2 text-sm text-red-600">{errors.items}</p>}
                            </div>
                        )}

                        {/* Total Summary */}
                        {data.items.length > 0 && (
                            <div className="rounded-xl bg-gradient-to-r from-green-50 to-emerald-50 border border-green-200 p-5 shadow-sm">
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center gap-2">
                                        <div className="flex h-9 w-9 items-center justify-center rounded-lg bg-green-600 text-white">
                                            <svg className="h-5 w-5" fill="none" viewBox="0 0 24 24" strokeWidth="1.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M2.25 18.75a60.07 60.07 0 0 1 15.797 2.101c.727.198 1.453-.342 1.453-1.096V18.75M3.75 4.5v.75A.75.75 0 0 1 3 6h-.75m0 0v-.375c0-.621.504-1.125 1.125-1.125H20.25M2.25 6v9m18-10.5v.75c0 .414.336.75.75.75h.75m-1.5-1.5h.375c.621 0 1.125.504 1.125 1.125v9.75c0 .621-.504 1.125-1.125 1.125h-.375m1.5-1.5H21a.75.75 0 0 0-.75.75v.75m0 0H3.75m0 0h-.375a1.125 1.125 0 0 1-1.125-1.125V15m1.5 1.5v-.75A.75.75 0 0 0 3 15h-.75M15 10.5a3 3 0 1 1-6 0 3 3 0 0 1 6 0Zm3 0h.008v.008H18V10.5Zm-12 0h.008v.008H6V10.5Z" /></svg>
                                        </div>
                                        <div>
                                            <p className="text-sm font-medium text-gray-700">Total Payment</p>
                                            <p className="text-xs text-gray-500">{data.items.length} shop{data.items.length !== 1 ? 's' : ''} selected</p>
                                        </div>
                                    </div>
                                    <p className="text-2xl font-bold text-green-800">{formatCurrency(totalAmount)}</p>
                                </div>
                            </div>
                        )}

                        {/* Step 3: Payment Details */}
                        {data.items.length > 0 && (
                            <div className="bg-white shadow-sm sm:rounded-lg p-6">
                                <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
                                    <span className="flex h-7 w-7 items-center justify-center rounded-full bg-blue-100 text-xs font-bold text-blue-700">3</span>
                                    Payment Details
                                </h3>

                                <div className="space-y-4">
                                    {/* Date & Method */}
                                    <div className="grid grid-cols-2 gap-4">
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">Payment Date <span className="text-red-500">*</span></label>
                                            <input
                                                type="date"
                                                value={data.payment_date}
                                                onChange={(e) => setData('payment_date', e.target.value)}
                                                className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                            />
                                            {errors.payment_date && <p className="mt-1 text-xs text-red-600">{errors.payment_date}</p>}
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">Payment Method <span className="text-red-500">*</span></label>
                                            <div className="grid grid-cols-2 gap-2">
                                                {[
                                                    { value: 'cash', label: 'Cash', icon: '💵' },
                                                    { value: 'bank_transfer', label: 'Bank Transfer', icon: '🏦' },
                                                ].map((method) => (
                                                    <button
                                                        key={method.value}
                                                        type="button"
                                                        onClick={() => setData('payment_method', method.value)}
                                                        className={`flex items-center justify-center gap-1.5 rounded-lg border-2 px-3 py-2.5 text-sm font-medium transition-all ${
                                                            data.payment_method === method.value
                                                                ? 'border-blue-500 bg-blue-50 text-blue-700 shadow-sm'
                                                                : 'border-gray-200 text-gray-600 hover:border-gray-300'
                                                        }`}
                                                    >
                                                        <span>{method.icon}</span>
                                                        <span>{method.label}</span>
                                                    </button>
                                                ))}
                                            </div>
                                            {errors.payment_method && <p className="mt-1 text-xs text-red-600">{errors.payment_method}</p>}
                                        </div>
                                    </div>

                                    {/* Bank proof */}
                                    {data.payment_method === 'bank_transfer' && (
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 mb-1">Proof of Payment</label>
                                            <input
                                                type="file"
                                                onChange={(e) => setData('proof_file', e.target.files[0])}
                                                className="w-full rounded-lg border border-gray-300 text-sm file:mr-3 file:rounded-l-lg file:border-0 file:bg-gray-100 file:px-4 file:py-2 file:text-sm file:font-medium file:text-gray-700 hover:file:bg-gray-200"
                                                accept="image/*,.pdf"
                                            />
                                            {errors.proof_file && <p className="mt-1 text-xs text-red-600">{errors.proof_file}</p>}
                                        </div>
                                    )}

                                    {/* Cash - signature */}
                                    {data.payment_method === 'cash' && (
                                        <div className="rounded-lg border border-amber-200 bg-amber-50 p-4">
                                            <div className="flex items-center gap-3">
                                                <input
                                                    type="checkbox"
                                                    id="owner_signature_taken"
                                                    checked={data.owner_signature_taken}
                                                    onChange={(e) => setData('owner_signature_taken', e.target.checked)}
                                                    className="h-4 w-4 rounded border-gray-300 text-amber-600 focus:ring-amber-500"
                                                />
                                                <label htmlFor="owner_signature_taken" className="text-sm font-medium text-amber-800">
                                                    💰 Owner signature has been taken for cash payment
                                                </label>
                                            </div>
                                        </div>
                                    )}

                                    {/* Remarks */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-1">Remarks (optional)</label>
                                        <textarea
                                            value={data.remarks}
                                            onChange={(e) => setData('remarks', e.target.value)}
                                            rows={2}
                                            className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500 text-sm"
                                            placeholder="Any additional notes..."
                                        />
                                        {errors.remarks && <p className="mt-1 text-xs text-red-600">{errors.remarks}</p>}
                                    </div>
                                </div>

                                {/* Submit */}
                                <div className="flex items-center justify-end gap-3 mt-6 pt-4 border-t border-gray-100">
                                    <Link href={route('owner-payments.index')} className="rounded-lg border border-gray-300 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">
                                        Cancel
                                    </Link>
                                    <button
                                        type="submit"
                                        disabled={processing || data.items.length === 0}
                                        className="inline-flex items-center gap-2 rounded-lg bg-blue-600 px-5 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                                    >
                                        {processing ? (
                                            <>
                                                <svg className="h-4 w-4 animate-spin" fill="none" viewBox="0 0 24 24"><circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" /><path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4z" /></svg>
                                                Processing...
                                            </>
                                        ) : (
                                            <>
                                                <svg className="h-4 w-4" fill="none" viewBox="0 0 24 24" strokeWidth="2" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M9 12.75 11.25 15 15 9.75M21 12a9 9 0 1 1-18 0 9 9 0 0 1 18 0Z" /></svg>
                                                Record Payment — {formatCurrency(totalAmount)}
                                            </>
                                        )}
                                    </button>
                                </div>
                            </div>
                        )}
                    </form>
                </div>
            </div>
        </AuthenticatedLayout>
    );
}
