import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import { Head, Link, useForm, router, usePage } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import FormModal from '@/Components/FormModal';
import ConfirmDeleteModal from '@/Components/ConfirmDeleteModal';
import { formatCurrency, formatDate, formatMonth } from '@/utils';

export default function Index({ readings, filters, leases, defaultRate }) {
    const { auth } = usePage().props;
    const [showAddModal, setShowAddModal] = useState(false);
    const [editingReading, setEditingReading] = useState(null);
    const [deleteId, setDeleteId] = useState(null);
    const [deleting, setDeleting] = useState(false);

    const addForm = useForm({
        lease_id: '',
        month: new Date().toISOString().slice(0, 7),
        previous_reading: '',
        current_reading: '',
        rate_per_unit: defaultRate || '',
    });

    const editForm = useForm({
        previous_reading: '',
        current_reading: '',
        rate_per_unit: '',
    });

    useEffect(() => {
        if (editingReading) {
            editForm.setData({
                previous_reading: editingReading.previous_reading || '',
                current_reading: editingReading.current_reading || '',
                rate_per_unit: editingReading.rate_per_unit || '',
            });
        }
    }, [editingReading]);

    const addUnits = (parseFloat(addForm.data.current_reading) || 0) - (parseFloat(addForm.data.previous_reading) || 0);
    const addAmount = addUnits > 0 ? addUnits * (parseFloat(addForm.data.rate_per_unit) || 0) : 0;

    const editUnits = (parseFloat(editForm.data.current_reading) || 0) - (parseFloat(editForm.data.previous_reading) || 0);
    const editAmount = editUnits > 0 ? editUnits * (parseFloat(editForm.data.rate_per_unit) || 0) : 0;

    const applyFilters = (newFilters) => {
        router.get(route('meter-readings.index'), { ...filters, ...newFilters }, { preserveState: true, replace: true });
    };

    const handleAdd = (e) => {
        e.preventDefault();
        addForm.post(route('meter-readings.store'), {
            onSuccess: () => {
                setShowAddModal(false);
                addForm.reset();
            },
        });
    };

    const handleEdit = (e) => {
        e.preventDefault();
        editForm.put(route('meter-readings.update', editingReading.id), {
            onSuccess: () => {
                setEditingReading(null);
            },
        });
    };

    const handleDelete = () => {
        setDeleting(true);
        router.delete(route('meter-readings.destroy', deleteId), {
            onSuccess: () => {
                setDeleteId(null);
                setDeleting(false);
            },
            onError: () => setDeleting(false),
        });
    };

    const CalcBox = ({ units, amount }) => (
        <div className="mt-4 rounded-lg bg-blue-50 border border-blue-200 p-4">
            <div className="flex items-center justify-between text-sm">
                <span className="text-blue-700 font-medium">Units Consumed:</span>
                <span className="text-blue-900 font-semibold">{units > 0 ? units : 0}</span>
            </div>
            <div className="flex items-center justify-between text-sm mt-1">
                <span className="text-blue-700 font-medium">Total Amount:</span>
                <span className="text-blue-900 font-semibold">{formatCurrency(amount)}</span>
            </div>
        </div>
    );

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Meter Readings</h2>}>
            <Head title="Meter Readings" />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-8">
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            {/* Header */}
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium">All Meter Readings</h3>
                                <button
                                    onClick={() => setShowAddModal(true)}
                                    className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                                >
                                    + Add Reading
                                </button>
                            </div>

                            {/* Filters */}
                            <div className="mb-6">
                                <input
                                    type="month"
                                    value={filters.month || ''}
                                    onChange={(e) => applyFilters({ month: e.target.value })}
                                    className="w-full max-w-xs rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                />
                            </div>

                            {/* Table */}
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Shop#</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Tenant</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Month</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium uppercase tracking-wider text-gray-500">Previous</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium uppercase tracking-wider text-gray-500">Current</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium uppercase tracking-wider text-gray-500">Units</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium uppercase tracking-wider text-gray-500">Rate</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium uppercase tracking-wider text-gray-500">Amount (PKR)</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 bg-white">
                                        {readings.data.map((reading) => (
                                            <tr key={reading.id} className="hover:bg-gray-50">
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-gray-900">{reading.lease?.shop?.shop_number || '—'}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-gray-900">{reading.lease?.tenant?.name || '—'}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-gray-500">{formatMonth(reading.month)}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-right text-gray-500">{reading.previous_reading}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-right text-gray-500">{reading.current_reading}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-right font-medium text-gray-900">{reading.units_consumed}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-right text-gray-500">{formatCurrency(reading.rate_per_unit)}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-right font-medium text-gray-900">{formatCurrency(reading.total_amount)}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm space-x-2">
                                                    {auth.user.is_admin && (
                                                    <button onClick={() => setEditingReading(reading)} className="text-blue-600 hover:text-blue-900">Edit</button>
                                                    )}
                                                    {auth.user.is_admin && (
                                                    <button onClick={() => setDeleteId(reading.id)} className="text-red-600 hover:text-red-900">Delete</button>
                                                    )}
                                                </td>
                                            </tr>
                                        ))}
                                        {readings.data.length === 0 && (
                                            <tr>
                                                <td colSpan="9" className="px-4 py-4 text-center text-sm text-gray-500">No meter readings found.</td>
                                            </tr>
                                        )}
                                    </tbody>
                                </table>
                            </div>

                            {/* Pagination */}
                            {readings.links && readings.links.length > 3 && (
                                <div className="flex flex-wrap items-center gap-1 mt-4">
                                    {readings.links.map((link, i) => (
                                        link.url ? (
                                            <Link key={i} href={link.url} className={`px-3 py-1 text-sm rounded ${link.active ? 'bg-blue-600 text-white' : 'bg-white border hover:bg-gray-50'}`}
                                                dangerouslySetInnerHTML={{ __html: link.label }} preserveState />
                                        ) : (
                                            <span key={i} className="px-3 py-1 text-sm text-gray-400" dangerouslySetInnerHTML={{ __html: link.label }} />
                                        )
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>

            {/* Add Modal */}
            <FormModal show={showAddModal} onClose={() => setShowAddModal(false)} title="Add Meter Reading">
                <form onSubmit={handleAdd} className="space-y-4">
                    <div>
                        <label className="block text-sm font-medium text-gray-700">Lease *</label>
                        <select
                            value={addForm.data.lease_id}
                            onChange={(e) => addForm.setData('lease_id', e.target.value)}
                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        >
                            <option value="">Select a lease...</option>
                            {leases.map((lease) => (
                                <option key={lease.id} value={lease.id}>
                                    Shop {lease.shop?.shop_number} - {lease.tenant?.name}
                                </option>
                            ))}
                        </select>
                        {addForm.errors.lease_id && <p className="mt-1 text-sm text-red-600">{addForm.errors.lease_id}</p>}
                    </div>

                    <div>
                        <label className="block text-sm font-medium text-gray-700">Month *</label>
                        <input
                            type="month"
                            value={addForm.data.month}
                            onChange={(e) => addForm.setData('month', e.target.value)}
                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        />
                        {addForm.errors.month && <p className="mt-1 text-sm text-red-600">{addForm.errors.month}</p>}
                    </div>

                    <div className="grid grid-cols-2 gap-4">
                        <div>
                            <label className="block text-sm font-medium text-gray-700">Previous Reading *</label>
                            <input
                                type="number"
                                value={addForm.data.previous_reading}
                                onChange={(e) => addForm.setData('previous_reading', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                            />
                            {addForm.errors.previous_reading && <p className="mt-1 text-sm text-red-600">{addForm.errors.previous_reading}</p>}
                        </div>
                        <div>
                            <label className="block text-sm font-medium text-gray-700">Current Reading *</label>
                            <input
                                type="number"
                                value={addForm.data.current_reading}
                                onChange={(e) => addForm.setData('current_reading', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                            />
                            {addForm.errors.current_reading && <p className="mt-1 text-sm text-red-600">{addForm.errors.current_reading}</p>}
                        </div>
                    </div>

                    <div>
                        <label className="block text-sm font-medium text-gray-700">Rate per Unit *</label>
                        <input
                            type="number"
                            value={addForm.data.rate_per_unit}
                            onChange={(e) => addForm.setData('rate_per_unit', e.target.value)}
                            className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                            step="any"
                        />
                        {addForm.errors.rate_per_unit && <p className="mt-1 text-sm text-red-600">{addForm.errors.rate_per_unit}</p>}
                    </div>

                    <CalcBox units={addUnits} amount={addAmount} />

                    <div className="flex justify-end gap-3 pt-4 border-t">
                        <button type="button" onClick={() => setShowAddModal(false)} className="rounded-md bg-white px-4 py-2 text-sm font-medium text-gray-700 border border-gray-300 hover:bg-gray-50">
                            Cancel
                        </button>
                        <button type="submit" disabled={addForm.processing} className="rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50">
                            {addForm.processing ? 'Saving...' : 'Save Reading'}
                        </button>
                    </div>
                </form>
            </FormModal>

            {/* Edit Modal */}
            <FormModal show={!!editingReading} onClose={() => setEditingReading(null)} title="Edit Meter Reading">
                {editingReading && (
                    <form onSubmit={handleEdit} className="space-y-4">
                        <div>
                            <label className="block text-sm font-medium text-gray-700">Lease</label>
                            <p className="mt-1 text-sm text-gray-900">
                                Shop {editingReading.lease?.shop?.shop_number} - {editingReading.lease?.tenant?.name}
                            </p>
                        </div>

                        <div>
                            <label className="block text-sm font-medium text-gray-700">Month</label>
                            <p className="mt-1 text-sm text-gray-900">{formatMonth(editingReading.month)}</p>
                        </div>

                        <div className="grid grid-cols-2 gap-4">
                            <div>
                                <label className="block text-sm font-medium text-gray-700">Previous Reading *</label>
                                <input
                                    type="number"
                                    value={editForm.data.previous_reading}
                                    onChange={(e) => editForm.setData('previous_reading', e.target.value)}
                                    className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                />
                                {editForm.errors.previous_reading && <p className="mt-1 text-sm text-red-600">{editForm.errors.previous_reading}</p>}
                            </div>
                            <div>
                                <label className="block text-sm font-medium text-gray-700">Current Reading *</label>
                                <input
                                    type="number"
                                    value={editForm.data.current_reading}
                                    onChange={(e) => editForm.setData('current_reading', e.target.value)}
                                    className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                />
                                {editForm.errors.current_reading && <p className="mt-1 text-sm text-red-600">{editForm.errors.current_reading}</p>}
                            </div>
                        </div>

                        <div>
                            <label className="block text-sm font-medium text-gray-700">Rate per Unit *</label>
                            <input
                                type="number"
                                value={editForm.data.rate_per_unit}
                                onChange={(e) => editForm.setData('rate_per_unit', e.target.value)}
                                className="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                step="any"
                            />
                            {editForm.errors.rate_per_unit && <p className="mt-1 text-sm text-red-600">{editForm.errors.rate_per_unit}</p>}
                        </div>

                        <CalcBox units={editUnits} amount={editAmount} />

                        <div className="flex justify-end gap-3 pt-4 border-t">
                            <button type="button" onClick={() => setEditingReading(null)} className="rounded-md bg-white px-4 py-2 text-sm font-medium text-gray-700 border border-gray-300 hover:bg-gray-50">
                                Cancel
                            </button>
                            <button type="submit" disabled={editForm.processing} className="rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50">
                                {editForm.processing ? 'Updating...' : 'Update Reading'}
                            </button>
                        </div>
                    </form>
                )}
            </FormModal>

            {/* Delete Confirmation */}
            <ConfirmDeleteModal
                show={!!deleteId}
                onClose={() => setDeleteId(null)}
                onConfirm={handleDelete}
                title="Delete Meter Reading"
                message="Are you sure you want to delete this meter reading? This action cannot be undone."
                processing={deleting}
            />
        </AuthenticatedLayout>
    );
}
