import AuthenticatedLayout from '@/Layouts/AuthenticatedLayout';
import FormModal from '@/Components/FormModal';
import ConfirmDeleteModal from '@/Components/ConfirmDeleteModal';
import { Head, Link, useForm, router, usePage } from '@inertiajs/react';
import { formatCurrency, formatDate } from '@/utils';
import { useState, useEffect, useMemo } from 'react';

export default function Index({ expenses, categories, totalIn, totalOut, balance, filters }) {
    const { auth } = usePage().props;
    const [deleteId, setDeleteId] = useState(null);
    const [deleting, setDeleting] = useState(false);
    const [showAddModal, setShowAddModal] = useState(false);
    const [editingExpense, setEditingExpense] = useState(null);

    const addForm = useForm({
        type: 'cash_out',
        category_id: '',
        amount: '',
        description: '',
        date: new Date().toISOString().split('T')[0],
        reference_no: '',
        proof_file: null,
    });

    const editForm = useForm({
        type: 'cash_out',
        category_id: '',
        amount: '',
        description: '',
        date: '',
        reference_no: '',
        proof_file: null,
    });

    useEffect(() => {
        if (editingExpense) {
            editForm.setData({
                type: editingExpense.type || 'cash_out',
                category_id: editingExpense.category_id || '',
                amount: editingExpense.amount || '',
                description: editingExpense.description || '',
                date: editingExpense.date || '',
                reference_no: editingExpense.reference_no || '',
                proof_file: null,
            });
        }
    }, [editingExpense]);

    const getFilteredCategories = (type) => {
        const categoryType = type === 'cash_in' ? 'income' : 'expense';
        return categories.filter(cat => cat.type === categoryType);
    };

    const addFilteredCategories = useMemo(() => getFilteredCategories(addForm.data.type), [addForm.data.type, categories]);
    const editFilteredCategories = useMemo(() => getFilteredCategories(editForm.data.type), [editForm.data.type, categories]);

    const handleTypeChange = (form, type) => {
        form.setData(prev => ({ ...prev, type, category_id: '' }));
    };

    const applyFilters = (newFilters) => {
        router.get(route('expenses.index'), { ...filters, ...newFilters }, { preserveState: true, replace: true });
    };

    const handleDelete = () => {
        setDeleting(true);
        router.delete(route('expenses.destroy', deleteId), {
            onFinish: () => { setDeleting(false); setDeleteId(null); },
        });
    };

    const handleAdd = (e) => {
        e.preventDefault();
        addForm.post(route('expenses.store'), {
            forceFormData: true,
            onSuccess: () => {
                setShowAddModal(false);
                addForm.reset();
                addForm.setData('date', new Date().toISOString().split('T')[0]);
            },
        });
    };

    const handleEdit = (e) => {
        e.preventDefault();
        editForm.transform((data) => ({ ...data, _method: 'put' })).post(route('expenses.update', editingExpense.id), {
            forceFormData: true,
            onSuccess: () => setEditingExpense(null),
        });
    };

    const renderFormFields = (form, filteredCategories) => (
        <>
            {/* Type Radio Cards */}
            <div className="space-y-1.5">
                <label className="block text-sm font-medium text-gray-700">Type <span className="text-red-500">*</span></label>
                <div className="grid grid-cols-2 gap-3">
                    <button
                        type="button"
                        onClick={() => handleTypeChange(form, 'cash_in')}
                        className={`flex items-center justify-center gap-2 rounded-lg border-2 px-4 py-3 text-sm font-medium transition-all ${
                            form.data.type === 'cash_in'
                                ? 'border-green-500 bg-green-50 text-green-700 ring-1 ring-green-500'
                                : 'border-gray-200 bg-white text-gray-600 hover:border-gray-300'
                        }`}
                    >
                        <svg className="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 4v16m0-16l-4 4m4-4l4 4" /></svg>
                        Cash In
                    </button>
                    <button
                        type="button"
                        onClick={() => handleTypeChange(form, 'cash_out')}
                        className={`flex items-center justify-center gap-2 rounded-lg border-2 px-4 py-3 text-sm font-medium transition-all ${
                            form.data.type === 'cash_out'
                                ? 'border-red-500 bg-red-50 text-red-700 ring-1 ring-red-500'
                                : 'border-gray-200 bg-white text-gray-600 hover:border-gray-300'
                        }`}
                    >
                        <svg className="h-4 w-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M12 20V4m0 16l-4-4m4 4l4-4" /></svg>
                        Cash Out
                    </button>
                </div>
                {form.errors.type && <p className="text-sm text-red-600">{form.errors.type}</p>}
            </div>

            {/* Category */}
            <div className="space-y-1.5">
                <label className="block text-sm font-medium text-gray-700">Category <span className="text-red-500">*</span></label>
                <select
                    value={form.data.category_id}
                    onChange={(e) => form.setData('category_id', e.target.value)}
                    className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                >
                    <option value="">Select category</option>
                    {filteredCategories.map((cat) => (
                        <option key={cat.id} value={cat.id}>{cat.name}</option>
                    ))}
                </select>
                {form.errors.category_id && <p className="text-sm text-red-600">{form.errors.category_id}</p>}
            </div>

            {/* Amount + Date */}
            <div className="grid grid-cols-2 gap-4">
                <div className="space-y-1.5">
                    <label className="block text-sm font-medium text-gray-700">Amount <span className="text-red-500">*</span></label>
                    <input
                        type="number"
                        value={form.data.amount}
                        onChange={(e) => form.setData('amount', e.target.value)}
                        className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                        placeholder="0.00"
                        step="0.01"
                        min="0"
                    />
                    {form.errors.amount && <p className="text-sm text-red-600">{form.errors.amount}</p>}
                </div>
                <div className="space-y-1.5">
                    <label className="block text-sm font-medium text-gray-700">Date <span className="text-red-500">*</span></label>
                    <input
                        type="date"
                        value={form.data.date}
                        onChange={(e) => form.setData('date', e.target.value)}
                        className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                    />
                    {form.errors.date && <p className="text-sm text-red-600">{form.errors.date}</p>}
                </div>
            </div>

            {/* Description */}
            <div className="space-y-1.5">
                <label className="block text-sm font-medium text-gray-700">Description <span className="text-red-500">*</span></label>
                <textarea
                    value={form.data.description}
                    onChange={(e) => form.setData('description', e.target.value)}
                    rows={2}
                    className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                    placeholder="Enter description..."
                />
                {form.errors.description && <p className="text-sm text-red-600">{form.errors.description}</p>}
            </div>

            {/* Reference No */}
            <div className="space-y-1.5">
                <label className="block text-sm font-medium text-gray-700">Reference No</label>
                <input
                    type="text"
                    value={form.data.reference_no}
                    onChange={(e) => form.setData('reference_no', e.target.value)}
                    className="w-full rounded-lg border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                    placeholder="e.g. INV-001"
                />
                {form.errors.reference_no && <p className="text-sm text-red-600">{form.errors.reference_no}</p>}
            </div>

            {/* Proof File */}
            <div className="space-y-1.5">
                <label className="block text-sm font-medium text-gray-700">Proof File</label>
                <input
                    type="file"
                    onChange={(e) => form.setData('proof_file', e.target.files[0])}
                    className="w-full rounded-lg border border-gray-300 px-3 py-2 text-sm text-gray-700 file:mr-4 file:rounded-md file:border-0 file:bg-blue-50 file:px-3 file:py-1.5 file:text-sm file:font-medium file:text-blue-700 hover:file:bg-blue-100"
                />
                {form.errors.proof_file && <p className="text-sm text-red-600">{form.errors.proof_file}</p>}
            </div>
        </>
    );

    return (
        <AuthenticatedLayout header={<h2 className="text-xl font-semibold leading-tight text-gray-800">Cash Book</h2>}>
            <Head title="Cash Book" />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-8 space-y-6">
                    {/* Summary Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                        <div className="bg-white shadow-sm sm:rounded-lg p-6 border-l-4 border-green-500">
                            <p className="text-sm font-medium text-gray-500">Cash In</p>
                            <p className="text-2xl font-bold text-green-700">{formatCurrency(totalIn)}</p>
                        </div>
                        <div className="bg-white shadow-sm sm:rounded-lg p-6 border-l-4 border-red-500">
                            <p className="text-sm font-medium text-gray-500">Cash Out</p>
                            <p className="text-2xl font-bold text-red-700">{formatCurrency(totalOut)}</p>
                        </div>
                        <div className={`bg-white shadow-sm sm:rounded-lg p-6 border-l-4 ${balance >= 0 ? 'border-blue-500' : 'border-red-500'}`}>
                            <p className="text-sm font-medium text-gray-500">Balance</p>
                            <p className={`text-2xl font-bold ${balance >= 0 ? 'text-blue-700' : 'text-red-700'}`}>{formatCurrency(balance)}</p>
                        </div>
                    </div>

                    {/* Main Table */}
                    <div className="overflow-hidden bg-white shadow-sm sm:rounded-lg">
                        <div className="p-6 text-gray-900">
                            {/* Header */}
                            <div className="flex items-center justify-between mb-6">
                                <h3 className="text-lg font-medium">All Entries</h3>
                                <div className="flex items-center gap-3">
                                    {/* Quick Report Buttons */}
                                    <div className="flex items-center gap-2 border-r border-gray-300 pr-3">
                                        <Link
                                            href={route('expenses.report', { period: 'daily', date: new Date().toISOString().split('T')[0] })}
                                            className="inline-flex items-center gap-1.5 rounded-md bg-purple-50 px-3 py-2 text-sm font-medium text-purple-700 hover:bg-purple-100 transition-colors"
                                        >
                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" /></svg>
                                            Today
                                        </Link>
                                        <Link
                                            href={route('expenses.report', { period: 'weekly', date: new Date().toISOString().split('T')[0] })}
                                            className="inline-flex items-center gap-1.5 rounded-md bg-emerald-50 px-3 py-2 text-sm font-medium text-emerald-700 hover:bg-emerald-100 transition-colors"
                                        >
                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" /></svg>
                                            This Week
                                        </Link>
                                        <Link
                                            href={route('expenses.report', { period: 'monthly', date: new Date().toISOString().split('T')[0] })}
                                            className="inline-flex items-center gap-1.5 rounded-md bg-amber-50 px-3 py-2 text-sm font-medium text-amber-700 hover:bg-amber-100 transition-colors"
                                        >
                                            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" /></svg>
                                            This Month
                                        </Link>
                                    </div>
                                    <button
                                        onClick={() => setShowAddModal(true)}
                                        className="inline-flex items-center rounded-md bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700"
                                    >
                                        + Add Entry
                                    </button>
                                </div>
                            </div>

                            {/* Filters */}
                            <div className="mb-6 grid grid-cols-1 md:grid-cols-4 gap-4">
                                <select
                                    value={filters.type || ''}
                                    onChange={(e) => applyFilters({ type: e.target.value })}
                                    className="rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                >
                                    <option value="">All Types</option>
                                    <option value="cash_in">Cash In</option>
                                    <option value="cash_out">Cash Out</option>
                                </select>

                                <select
                                    value={filters.category_id || ''}
                                    onChange={(e) => applyFilters({ category_id: e.target.value })}
                                    className="rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                >
                                    <option value="">All Categories</option>
                                    {categories.map((cat) => (
                                        <option key={cat.id} value={cat.id}>{cat.name}</option>
                                    ))}
                                </select>

                                <input
                                    type="date"
                                    value={filters.date_from || ''}
                                    onChange={(e) => applyFilters({ date_from: e.target.value })}
                                    className="rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    placeholder="From"
                                />

                                <input
                                    type="date"
                                    value={filters.date_to || ''}
                                    onChange={(e) => applyFilters({ date_to: e.target.value })}
                                    className="rounded-md border-gray-300 shadow-sm focus:border-blue-500 focus:ring-blue-500"
                                    placeholder="To"
                                />
                            </div>

                            {/* Table */}
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Date</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Type</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Category</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Description</th>
                                            <th className="px-4 py-3 text-right text-xs font-medium uppercase tracking-wider text-gray-500">Amount</th>
                                            <th className="px-4 py-3 text-center text-xs font-medium uppercase tracking-wider text-gray-500">Proof</th>
                                            <th className="px-4 py-3 text-left text-xs font-medium uppercase tracking-wider text-gray-500">Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody className="divide-y divide-gray-200 bg-white">
                                        {expenses.data.map((expense) => (
                                            <tr key={expense.id} className="hover:bg-gray-50">
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-gray-500">{formatDate(expense.date)}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm">
                                                    <span className={`inline-flex items-center rounded-full px-2.5 py-0.5 text-xs font-medium ${
                                                        expense.type === 'cash_in'
                                                            ? 'bg-green-100 text-green-800'
                                                            : 'bg-red-100 text-red-800'
                                                    }`}>
                                                        {expense.type === 'cash_in' ? 'Cash In' : 'Cash Out'}
                                                    </span>
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-gray-500">{expense.category?.name || '—'}</td>
                                                <td className="max-w-xs truncate px-4 py-4 text-sm text-gray-900">{expense.description || '—'}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-right font-medium text-gray-900">{formatCurrency(expense.amount)}</td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm text-center">
                                                    {expense.proof_file ? (
                                                        <a href={`/storage/${expense.proof_file}`} target="_blank" rel="noopener noreferrer" className="text-green-600 hover:text-green-900">✓</a>
                                                    ) : (
                                                        <span className="text-gray-400">✗</span>
                                                    )}
                                                </td>
                                                <td className="whitespace-nowrap px-4 py-4 text-sm space-x-2">
                                                    {auth.user.is_admin && (
                                                    <button onClick={() => setEditingExpense(expense)} className="text-blue-600 hover:text-blue-900">Edit</button>
                                                    )}
                                                    {auth.user.is_admin && (
                                                    <button onClick={() => setDeleteId(expense.id)} className="text-red-600 hover:text-red-900">Delete</button>
                                                    )}
                                                </td>
                                            </tr>
                                        ))}
                                        {expenses.data.length === 0 && (
                                            <tr>
                                                <td colSpan="7" className="px-4 py-4 text-center text-sm text-gray-500">No entries found.</td>
                                            </tr>
                                        )}
                                    </tbody>
                                </table>
                            </div>

                            {/* Pagination */}
                            {expenses.links && expenses.links.length > 3 && (
                                <div className="flex flex-wrap items-center gap-1 mt-4">
                                    {expenses.links.map((link, i) => (
                                        link.url ? (
                                            <Link key={i} href={link.url} className={`px-3 py-1 text-sm rounded ${link.active ? 'bg-blue-600 text-white' : 'bg-white border hover:bg-gray-50'}`}
                                                dangerouslySetInnerHTML={{ __html: link.label }} preserveState />
                                        ) : (
                                            <span key={i} className="px-3 py-1 text-sm text-gray-400" dangerouslySetInnerHTML={{ __html: link.label }} />
                                        )
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>

            {/* Add Modal */}
            <FormModal show={showAddModal} onClose={() => { setShowAddModal(false); addForm.clearErrors(); }} title="Add Entry">
                <form onSubmit={handleAdd} className="space-y-4">
                    {renderFormFields(addForm, addFilteredCategories)}
                    <div className="flex justify-end gap-3 border-t border-gray-100 pt-4">
                        <button type="button" onClick={() => { setShowAddModal(false); addForm.clearErrors(); }} className="rounded-lg border border-gray-300 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">
                            Cancel
                        </button>
                        <button type="submit" disabled={addForm.processing} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50">
                            {addForm.processing ? 'Saving...' : 'Save Entry'}
                        </button>
                    </div>
                </form>
            </FormModal>

            {/* Edit Modal */}
            <FormModal show={editingExpense !== null} onClose={() => { setEditingExpense(null); editForm.clearErrors(); }} title="Edit Entry">
                <form onSubmit={handleEdit} className="space-y-4">
                    {renderFormFields(editForm, editFilteredCategories)}
                    <div className="flex justify-end gap-3 border-t border-gray-100 pt-4">
                        <button type="button" onClick={() => { setEditingExpense(null); editForm.clearErrors(); }} className="rounded-lg border border-gray-300 px-4 py-2 text-sm font-medium text-gray-700 hover:bg-gray-50">
                            Cancel
                        </button>
                        <button type="submit" disabled={editForm.processing} className="rounded-lg bg-blue-600 px-4 py-2 text-sm font-medium text-white hover:bg-blue-700 disabled:opacity-50">
                            {editForm.processing ? 'Updating...' : 'Update Entry'}
                        </button>
                    </div>
                </form>
            </FormModal>

            <ConfirmDeleteModal
                show={deleteId !== null}
                onClose={() => setDeleteId(null)}
                onConfirm={handleDelete}
                title="Delete Entry"
                message="Are you sure you want to delete this cash book entry?"
                processing={deleting}
            />
        </AuthenticatedLayout>
    );
}
