import { useState, useRef, useEffect } from 'react';

export default function SearchableSelect({ options = [], value, onChange, placeholder = 'Search...', error, disabled = false }) {
    const [isOpen, setIsOpen] = useState(false);
    const [search, setSearch] = useState('');
    const [highlightIndex, setHighlightIndex] = useState(-1);
    const containerRef = useRef(null);
    const inputRef = useRef(null);
    const listRef = useRef(null);

    const selectedOption = options.find(o => String(o.value) === String(value));

    const filtered = options.filter(o =>
        o.label.toLowerCase().includes(search.toLowerCase())
    );

    useEffect(() => {
        const handleClickOutside = (e) => {
            if (containerRef.current && !containerRef.current.contains(e.target)) {
                setIsOpen(false);
                setSearch('');
                setHighlightIndex(-1);
            }
        };
        document.addEventListener('mousedown', handleClickOutside);
        return () => document.removeEventListener('mousedown', handleClickOutside);
    }, []);

    useEffect(() => {
        if (isOpen && listRef.current && highlightIndex >= 0) {
            const el = listRef.current.children[highlightIndex];
            if (el) el.scrollIntoView({ block: 'nearest' });
        }
    }, [highlightIndex]);

    const handleSelect = (option) => {
        onChange(String(option.value));
        setIsOpen(false);
        setSearch('');
        setHighlightIndex(-1);
    };

    const handleKeyDown = (e) => {
        if (!isOpen) {
            if (e.key === 'ArrowDown' || e.key === 'Enter') {
                setIsOpen(true);
                e.preventDefault();
            }
            return;
        }

        switch (e.key) {
            case 'ArrowDown':
                e.preventDefault();
                setHighlightIndex(prev => Math.min(prev + 1, filtered.length - 1));
                break;
            case 'ArrowUp':
                e.preventDefault();
                setHighlightIndex(prev => Math.max(prev - 1, 0));
                break;
            case 'Enter':
                e.preventDefault();
                if (highlightIndex >= 0 && filtered[highlightIndex]) {
                    handleSelect(filtered[highlightIndex]);
                }
                break;
            case 'Escape':
                setIsOpen(false);
                setSearch('');
                setHighlightIndex(-1);
                break;
        }
    };

    const handleClear = (e) => {
        e.stopPropagation();
        onChange('');
        setSearch('');
        setIsOpen(false);
    };

    return (
        <div ref={containerRef} className="relative">
            {/* Selected / Input */}
            <div
                onClick={() => {
                    if (!disabled) {
                        setIsOpen(true);
                        setTimeout(() => inputRef.current?.focus(), 0);
                    }
                }}
                className={`mt-1 flex items-center gap-2 w-full rounded-md border shadow-sm px-3 py-2 cursor-pointer transition-all ${
                    disabled
                        ? 'bg-gray-100 border-gray-300 cursor-not-allowed'
                        : isOpen
                            ? 'border-blue-500 ring-1 ring-blue-500'
                            : error
                                ? 'border-red-300'
                                : 'border-gray-300 hover:border-gray-400'
                }`}
            >
                {isOpen ? (
                    <input
                        ref={inputRef}
                        type="text"
                        value={search}
                        onChange={(e) => { setSearch(e.target.value); setHighlightIndex(0); }}
                        onKeyDown={handleKeyDown}
                        placeholder={selectedOption ? selectedOption.label : placeholder}
                        className="flex-1 border-0 p-0 text-sm focus:ring-0 focus:outline-none bg-transparent placeholder-gray-400"
                        autoFocus
                    />
                ) : (
                    <span className={`flex-1 text-sm truncate ${selectedOption ? 'text-gray-900' : 'text-gray-400'}`}>
                        {selectedOption ? selectedOption.label : placeholder}
                    </span>
                )}

                <div className="flex items-center gap-1 flex-shrink-0">
                    {selectedOption && !disabled && (
                        <button type="button" onClick={handleClear} className="text-gray-400 hover:text-gray-600">
                            <svg className="h-4 w-4" fill="none" viewBox="0 0 24 24" strokeWidth="2" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="M6 18 18 6M6 6l12 12" /></svg>
                        </button>
                    )}
                    <svg className={`h-4 w-4 text-gray-400 transition-transform ${isOpen ? 'rotate-180' : ''}`} fill="none" viewBox="0 0 24 24" strokeWidth="2" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="m19.5 8.25-7.5 7.5-7.5-7.5" /></svg>
                </div>
            </div>

            {/* Dropdown */}
            {isOpen && (
                <div className="absolute z-50 mt-1 w-full rounded-md bg-white shadow-lg border border-gray-200 max-h-60 overflow-auto" ref={listRef}>
                    {filtered.length === 0 ? (
                        <div className="px-4 py-3 text-sm text-gray-500 text-center">No results found</div>
                    ) : (
                        filtered.map((option, idx) => (
                            <div
                                key={option.value}
                                onClick={() => handleSelect(option)}
                                className={`flex items-center gap-3 px-3 py-2.5 cursor-pointer transition-colors border-b border-gray-50 last:border-0 ${
                                    idx === highlightIndex
                                        ? 'bg-blue-50'
                                        : String(option.value) === String(value)
                                            ? 'bg-blue-50/50'
                                            : 'hover:bg-gray-50'
                                }`}
                            >
                                {/* Icon */}
                                <div className={`flex-shrink-0 h-9 min-w-9 px-2 rounded-lg flex items-center justify-center text-xs font-semibold ${
                                    String(option.value) === String(value)
                                        ? 'bg-blue-100 text-blue-700'
                                        : 'bg-gray-100 text-gray-600'
                                }`}>
                                    {option.icon || option.label.charAt(0).toUpperCase()}
                                </div>
                                <div className="flex-1 min-w-0">
                                    <p className={`text-sm font-medium truncate ${String(option.value) === String(value) ? 'text-blue-700' : 'text-gray-900'}`}>
                                        {option.label}
                                    </p>
                                    {option.subtitle && (
                                        <p className="text-xs text-gray-500 truncate">{option.subtitle}</p>
                                    )}
                                </div>
                                {String(option.value) === String(value) && (
                                    <svg className="h-4 w-4 text-blue-600 flex-shrink-0" fill="none" viewBox="0 0 24 24" strokeWidth="2.5" stroke="currentColor"><path strokeLinecap="round" strokeLinejoin="round" d="m4.5 12.75 6 6 9-13.5" /></svg>
                                )}
                            </div>
                        ))
                    )}
                </div>
            )}
        </div>
    );
}
