import { useState, useEffect, useRef } from 'react';

export default function ConfirmDeleteModal({ show, onClose, onConfirm, title, message, processing }) {
    const [visible, setVisible] = useState(false);
    const cancelRef = useRef(null);

    useEffect(() => {
        if (show) {
            requestAnimationFrame(() => setVisible(true));
            cancelRef.current?.focus();
        } else {
            setVisible(false);
        }
    }, [show]);

    if (!show) return null;

    const handleClose = () => {
        setVisible(false);
        setTimeout(() => onClose(), 200);
    };

    return (
        <div className="fixed inset-0 z-[9998] overflow-y-auto">
            {/* Backdrop */}
            <div
                className={`fixed inset-0 bg-black transition-opacity duration-200 ${visible ? 'bg-opacity-40' : 'bg-opacity-0'}`}
                onClick={handleClose}
            />

            {/* Modal */}
            <div className="flex min-h-full items-center justify-center p-4">
                <div
                    className={`relative w-full max-w-md transform rounded-xl bg-white shadow-2xl transition-all duration-200 ${
                        visible ? 'scale-100 opacity-100' : 'scale-95 opacity-0'
                    }`}
                >
                    <div className="p-6">
                        {/* Icon */}
                        <div className="mx-auto flex h-14 w-14 items-center justify-center rounded-full bg-red-100 mb-4">
                            <svg className="h-7 w-7 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                    strokeWidth="2"
                                    d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z"
                                />
                            </svg>
                        </div>

                        {/* Content */}
                        <div className="text-center">
                            <h3 className="text-lg font-semibold text-gray-900">{title || 'Delete Confirmation'}</h3>
                            <p className="mt-2 text-sm text-gray-500">
                                {message || 'Are you sure you want to delete this? This action cannot be undone.'}
                            </p>
                        </div>

                        {/* Actions */}
                        <div className="mt-6 flex gap-3">
                            <button
                                ref={cancelRef}
                                onClick={handleClose}
                                disabled={processing}
                                className="flex-1 rounded-lg border border-gray-300 bg-white px-4 py-2.5 text-sm font-medium text-gray-700 hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-gray-300 transition-colors"
                            >
                                Cancel
                            </button>
                            <button
                                onClick={onConfirm}
                                disabled={processing}
                                className="flex-1 rounded-lg bg-red-600 px-4 py-2.5 text-sm font-medium text-white hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-red-500 disabled:opacity-50 transition-colors"
                            >
                                {processing ? (
                                    <span className="flex items-center justify-center gap-2">
                                        <svg className="h-4 w-4 animate-spin" fill="none" viewBox="0 0 24 24">
                                            <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4" />
                                            <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8v4a4 4 0 00-4 4H4z" />
                                        </svg>
                                        Deleting...
                                    </span>
                                ) : (
                                    'Yes, Delete'
                                )}
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
}
