<?php

namespace App\Http\Controllers;

use App\Models\Owner;
use App\Models\Shop;
use Illuminate\Http\Request;
use Inertia\Inertia;

class ShopController extends Controller
{
    public function index(Request $request)
    {
        $shops = Shop::with(['owner', 'activeLease.tenant'])
            ->when($request->search, fn($q, $s) => $q->where('shop_number', 'like', "%{$s}%"))
            ->when($request->owner_id, fn($q, $id) => $q->where('owner_id', $id))
            ->when($request->status, fn($q, $s) => $q->where('status', $s))
            ->latest()
            ->paginate(20)
            ->withQueryString();

        $owners = Owner::orderBy('name')->get(['id', 'name']);

        return Inertia::render('Shops/Index', [
            'shops' => $shops,
            'owners' => $owners,
            'filters' => $request->only('search', 'owner_id', 'status'),
        ]);
    }

    public function create()
    {
        $owners = Owner::orderBy('name')->get(['id', 'name']);
        return Inertia::render('Shops/Create', ['owners' => $owners]);
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'owner_id' => 'required|exists:owners,id',
            'shop_number' => 'required|string|max:50',
            'description' => 'nullable|string',
            'location' => 'nullable|string|max:255',
            'area_sqft' => 'nullable|numeric|min:0',
            'default_rent' => 'required|numeric|min:0',
        ]);

        Shop::create($validated);

        return redirect()->route('shops.index')->with('success', 'Shop created successfully.');
    }

    public function show(Shop $shop)
    {
        $shop->load(['owner', 'leases.tenant', 'activeLease.tenant']);

        return Inertia::render('Shops/Show', ['shop' => $shop]);
    }

    public function edit(Shop $shop)
    {
        $owners = Owner::orderBy('name')->get(['id', 'name']);
        return Inertia::render('Shops/Edit', ['shop' => $shop, 'owners' => $owners]);
    }

    public function update(Request $request, Shop $shop)
    {
        $validated = $request->validate([
            'owner_id' => 'required|exists:owners,id',
            'shop_number' => 'required|string|max:50',
            'description' => 'nullable|string',
            'location' => 'nullable|string|max:255',
            'area_sqft' => 'nullable|numeric|min:0',
            'default_rent' => 'required|numeric|min:0',
        ]);

        $shop->update($validated);

        return redirect()->route('shops.index')->with('success', 'Shop updated successfully.');
    }

    public function destroy(Shop $shop)
    {
        $shop->delete();
        return redirect()->route('shops.index')->with('success', 'Shop deleted successfully.');
    }
}
