<?php

namespace App\Http\Controllers;

use App\Models\ExpenseCategory;
use App\Models\Setting;
use Illuminate\Http\Request;
use Inertia\Inertia;

class SettingController extends Controller
{
    public function index()
    {
        $settings = Setting::all()->pluck('value', 'key');
        $categories = ExpenseCategory::orderBy('type')->orderBy('name')->get();

        return Inertia::render('Settings/Index', [
            'settings' => $settings,
            'categories' => $categories,
        ]);
    }

    public function update(Request $request)
    {
        $validated = $request->validate([
            'business_name' => 'nullable|string|max:255',
            'business_address' => 'nullable|string',
            'business_phone' => 'nullable|string|max:20',
            'default_electricity_rate' => 'nullable|numeric|min:0',
            'invoice_prefix' => 'nullable|string|max:10',
            'voucher_prefix' => 'nullable|string|max:10',
        ]);

        foreach ($validated as $key => $value) {
            Setting::set($key, $value);
        }

        if ($request->hasFile('business_logo')) {
            $path = $request->file('business_logo')->store('settings', 'public');
            Setting::set('business_logo', $path);
        }

        if ($request->hasFile('favicon')) {
            $path = $request->file('favicon')->store('settings', 'public');
            Setting::set('favicon', $path);
        }

        return redirect()->route('settings.index')->with('success', 'Settings updated.');
    }

    public function storeCategory(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:expense_categories,name',
            'type' => 'required|in:income,expense',
        ]);

        ExpenseCategory::create($validated);

        return redirect()->route('settings.index')->with('success', 'Category added successfully.');
    }

    public function updateCategory(Request $request, ExpenseCategory $category)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255|unique:expense_categories,name,' . $category->id,
            'type' => 'required|in:income,expense',
        ]);

        $category->update($validated);

        return redirect()->route('settings.index')->with('success', 'Category updated successfully.');
    }

    public function destroyCategory(ExpenseCategory $category)
    {
        if ($category->expenses()->count() > 0) {
            return redirect()->route('settings.index')->with('error', 'Cannot delete category with existing expenses.');
        }

        $category->delete();

        return redirect()->route('settings.index')->with('success', 'Category deleted successfully.');
    }
}
